class
	PATH_DISTRIBUTER

create
	make

feature {PATH_DISTRIBUTER} -- Access
	directory_collection: DIRECTORY_COLLECTION
	cd_collection: CD_COLLECTION
	medium_size: REAL -- in mb
	excluded_directories: DIRECTORY_COLLECTION
	
feature -- Initialization
	make ( a_path_list : LIST[STRING]; a_medium_size: REAL ) is
			-- loads all directories in directory_collection;
			-- non-existing directories and those bigger than medium size will be excluded.
		require
			a_path_list_exists: a_path_list /= void and then a_path_list.count > 0
			medium_size_positive: a_medium_size > 0
		do
			medium_size := a_medium_size
			create directory_collection.from_path_list ( a_path_list )
			create excluded_directories.make
			excluded_directories := directory_collection.exclude_bigger_than ( medium_size )
		ensure
			medium_size_set: medium_size = a_medium_size
		end

feature -- Public functions
	sort : CD_COLLECTION is
		-- sort directories on cds, return a cd_collection;
		-- directories from directory_collection remain the same
		require
		local
			a_number_of_cds : INTEGER
		do
			if ( directory_collection.count = 0 ) then
				-- make an empty cd:
				create Result.with_number_of_items ( 1, medium_size )
			else
				if ( directory_collection.sum_of_sizes / medium_size - ( directory_collection.sum_of_sizes / medium_size ).truncated_to_integer.to_real  = 0) then
					-- there's no rounding error:
					a_number_of_cds := (directory_collection.sum_of_sizes / medium_size).truncated_to_integer
				else
					-- there's a rounding error, so round up:
					a_number_of_cds := ( directory_collection.sum_of_sizes / medium_size ).truncated_to_integer + 1
				end
			
				from
				until
					Result /= void or a_number_of_cds > directory_collection.count 
				loop
					Result := sort_with_desired_number_of_cds ( a_number_of_cds )
					if ( Result = void ) then
						-- if there's not enough space for all the directories, try again with a bigger number of cds
						a_number_of_cds := a_number_of_cds + 1
					end
				end
			end
			cd_collection := Result
		ensure
			Result_exists: Result /= void
			cd_collection_set: cd_collection = Result
		--	number_of_cds_smaller_equal_to_number_of_directories: a_number_of_cds <= directory_collection.count
		end
	
	output ( a_path_to_output_file: STRING ) is
		-- creates output file and puts result of the sort routine in it
		require
			output_path_exists: a_path_to_output_file /= void and then a_path_to_output_file.count > 0
		local
			output_file : FILE
		do
			output_file.make_create_read_write ( a_path_to_output_file )
			output_file.put_string ( cd_collection.out )
			output_file.put_string ( "Excluded directories:%N" )
			output_file.put_string ( excluded_directories.out )
		end

feature {PATH_DISTRIBUTER}-- Private functions
	sort_with_desired_number_of_cds ( a_desired_number_of_cds : INTEGER ) : CD_COLLECTION is
		-- makes backups of directory_collection; tries to sort cds with the given number;
		-- if there's not enough space for all the directories on a_desired_number_of_cds, return void linked list
		require
			a_desired_number_of_cds_not_negative: a_desired_number_of_cds >= 0
		local 
			directory_backup : DIRECTORY_COLLECTION
			there_were_changes: BOOLEAN
			unfinished_cd_collection: CD_COLLECTION
		do
			create unfinished_cd_collection.with_number_of_items ( a_desired_number_of_cds, medium_size )			
			create directory_backup.make
			create Result.make
			
			directory_backup.copy ( directory_collection ) 

			from 
				there_were_changes := true
			until ( not there_were_changes ) 
			loop		
				-- put on all the cds the most appropriate directory
				-- if there are still some directories left, but they could be sorted on cds then
				-- return void result.
				there_were_changes := false
				-- sort cds on criterium of number of directories -> cds with least directories should
				-- have the biggest choice:
				unfinished_cd_collection.sort_after_number_of_directories 
				-- Distribute directories over CDs:
				from
					unfinished_cd_collection.start
				until
					unfinished_cd_collection.after
				loop
					if ( directory_backup.find_next_directory_to_size ( unfinished_cd_collection.item.free_space ) > 0 ) then
						unfinished_cd_collection.item.add_directory ( directory_backup.item )
						directory_backup.remove
						directory_backup.back			
						there_were_changes := true
					else
						-- there's no directory smaller or equal to free space on cd.
						-- pack the cd and proceed with the rest.
						Result.extend ( unfinished_cd_collection.item )
						unfinished_cd_collection.remove
						unfinished_cd_collection.back
					end
					unfinished_cd_collection.forth
				end
			end

			if ( directory_backup.count > 0 ) then
				Result := void
			end
		ensure
			directory_collection_same: directory_collection.is_equal ( old directory_collection )
		end

		
end -- class PATH_DISTRIBUTER
