indexing

	description: "A timeline"
	status:	"See notice at end of class"
	author: "Till G. Bay"
	date: "$Date: 2004/10/20 10:03:03 $"
	revision: "$Revision: 1.4 $"

class TIMELINE inherit

	ARRAYED_LIST [TIMEABLE]
		rename
			make as container_make, 
			start as container_start
		redefine
			has, extend, wipe_out, prune
		end

create
	
	make

feature -- Initialisation

	make is
			-- Create timeline.
		do
			container_make (10)
			create suspended_objects.make (0)
			create delta.make_by_fine_seconds (0)
			create stop_actions
		ensure
			suspended_objects_exists: suspended_objects /= Void
			delta_exists: delta /= Void
			stop_actions_exists: stop_actions /= Void
		end
		
feature -- Access

	canvas: CANVAS
			-- Reference to the widget

	time: INTEGER
			-- Current time

	start_time: INTEGER
			-- Start time

feature -- Status report

	running : BOOLEAN is
			-- Is the animation running?
		do
			Result := (animation_agent /= Void)
		end

	has (t: TIMEABLE): BOOLEAN is
			-- Is `t' registered?
		do
			Result := Precursor (t) or is_suspended (t)
		end

	can_register (t: TIMEABLE): BOOLEAN is
			-- Can `t' be registered?
		do
			Result := t /= Void and then t.is_ready
		end

	is_suspended (t: TIMEABLE): BOOLEAN is
			-- Is `t' suspended?
		do
			Result := suspended_objects.has (t)
		end
			
	has_canvas: BOOLEAN is
			-- Has a canvas been set?
		do
			Result := (canvas /= Void)
		end
		
feature -- Status setting

	set_start_time (t: INTEGER) is
			-- Set start time to `t'.
		require
			not_running: not running
		do
			start_time := t
			time := start_time
		ensure
			start_time_set: start_time = t
			time_reset: time = start_time
		end
		
	reset is
			-- Reset timeline.
		local
			i: INTEGER
			t: TIMEABLE
		do
			time := start_time
			from
				i := 1
			until
				i > count
			loop
				i_th (i).reset
				i := i + 1
			end

			refreshed := False
			from
				suspended_objects.start
			until
				suspended_objects.is_empty or else suspended_objects.after
			loop
				t := suspended_objects.item
				t.reset
				suspended_objects.remove
				extend (t)
				if not suspended_objects.after then
					suspended_objects.forth				
				end
			end
		ensure
			time_reset: time = start_time
			no_suspended_objects: suspended_objects.is_empty
		end
		
	suspend (t: TIMEABLE) is
			-- Suspend `t'.
		require
			timeable_exists: t /= Void
			not_suspended: not is_suspended (t)
		do
			container_start
			prune (t)
			suspended_objects.extend (t)
			if is_empty and running then
				stop
			end
		ensure
			suspended: running implies is_suspended (t)
		end
		
	set_canvas (c: CANVAS) is
			-- Set canvas to `c'.
		require
			canvas_exists: c /= Void
		do
			canvas:= c
		ensure
			canvas_set: canvas = c
		end
		
	set_delay (d: DOUBLE) is
			-- Set animation delay to `d' seconds.
		require
			delay_positive: d >= 0
		do
			create delta.make_by_fine_seconds (d)
		ensure
			delta_exists: delta /= Void
		end

	set_timeout (t: EV_TIMEOUT) is
			-- Set timeout to `t'.
		require
			enabled: t /= Void implies t.interval > 0
		do
			timeout := t
		ensure
			timeout_set: timeout = t
		end
	
feature -- Element change

	extend (t: TIMEABLE) is
			-- Register `t'.
		do
			Precursor (t)
			t.set_timeline (Current)
		ensure then
			attached: t.has_timeline
		end

	add_stop_action (a: PROCEDURE [ANY, TUPLE]) is
			-- Add stop action `a'.
		require
			action_exists: a /= Void
		do
			stop_actions.extend (a)
		ensure
			added: stop_actions.has (a)
		end

feature -- Removal

	prune (t: TIMEABLE) is
			-- Unregister `t'.
		do
			if is_suspended (t) then
				suspended_objects.start
				suspended_objects.prune (t)
			else
				Precursor (t)
			end
		end

	wipe_out is
			-- Unregister all objects.
		do
			Precursor
			suspended_objects.wipe_out
		end

feature -- Basic operations

	start is
			-- Start the animation.
		require
			not_running: not running
		do
			if not is_empty then
				animation_agent := agent play
				if timeout /= Void then
					timeout.actions.extend (animation_agent)
				else
					(create {EV_ENVIRONMENT}).application.idle_actions.extend 
						(animation_agent)
				end
			end
		ensure
			running: not is_empty implies running
		end
		
	stop is
			-- Stop the animation.
		require
			running: running
		local
			app: EV_APPLICATION
		do
			if timeout /= Void then
				timeout.actions.start
				timeout.actions.prune (animation_agent)				
			else
				app := (create {EV_ENVIRONMENT}).application
				app.idle_actions.start
				app.idle_actions.prune (animation_agent)
			end
			animation_agent := Void
			stop_actions.call ([])
			stop_actions.wipe_out
		ensure
			not_running: not running
		end
	
feature {TIMEABLE} -- Basic commands

	refresh is
			-- Refresh canvas.
		require
			has_canvas: has_canvas
		do
			if not refreshed then
				canvas.refresh
				refreshed := True
			end
		end
		
feature {NONE} -- Implementation

	animation_agent: PROCEDURE [ANY, TUPLE]
			-- Agent for the animation

	suspended_objects: ARRAYED_LIST [TIMEABLE]
			-- List of objects that have been suspended
			
	stop_actions: ACTION_SEQUENCE [TUPLE]
			-- Actions executed when animation is stopped
			
	timeout: EV_TIMEOUT
			-- Timeout for the animation
			
	last_time: TIME
			-- The last time play has been called needed for time increase test

	old_count: INTEGER
			-- Old active timeable object count
			
	delta: TIME_DURATION
			-- The delta time that elapses before time is increased
	
	refreshed: BOOLEAN
			-- Has the canvas already been refreshed in current animation
			-- cycle?
			
	play is
			-- Do the animation playing.
		local
			current_time: TIME
			t: TIMEABLE
			i: INTEGER
		do
			create current_time.make_now
			if last_time = Void then
				last_time := current_time
			end

			if last_time + delta < current_time then
				refreshed := False
				from
					i := 1
				until
					i > count
				loop
					old_count := count
					t := i_th (i)
					if time >= t.start_time then
						t.execute_timed_action (time - t.start_time)
					end

					if count = old_count then
						i := i + 1
					end
				end
				last_time := current_time
				time := time + 1
			end
			
			if is_empty and running then
				stop
			end
		end

invariant
	
	suspended_objects_exist: suspended_objects /= Void
	delta_exists: delta /= Void
	stop_actions_exist: stop_actions /= Void
	running_definition: running = (animation_agent /= Void)
	has_canvas_definition: has_canvas = (canvas /= Void)

end

--|--------------------------------------------------------
--| This file is Copyright (C) 2003 by ETH Zurich.
--|
--| For questions, comments, additions or suggestions on
--| how to improve this package, please write to:
--|
--|     Till G. Bay <tillbay@student.ethz.ch>
--|
--|--------------------------------------------------------
