indexing
	description: 
		"[
			Displayers for elements of a city such as place_displayer, link_displayer.
		 ]"
		
	author: "Marcel Kessler & Michela Pedroni, ETH Zurich"
	date: "$Date: 2004/10/20 10:03:04 $"
	revision: "$Revision: 1.4 $"

deferred class ELEMENT_DISPLAYER inherit
	
	SHARED_DISPLAYER

	DISPLAYER
		redefine 
			make 
		end
	
feature -- Initialization

	make (an_element: like element) is
			-- Create displayer for `an_element'.
		do
			Precursor (an_element)
			create basic_representation.make
			create highlighted_representation.make
			create marked_representation.make
		ensure then
			basic_representation_exists: basic_representation /= Void
			highlighted_representation_exists: 
					highlighted_representation /= Void
			marked_representation_exists: marked_representation /= Void
		end

feature -- Status report

	highlighted: BOOLEAN
			-- Is the element highlighted?
	
	marked: BOOLEAN
			-- Is the element marked?
	
	shown: BOOLEAN
			-- Is the element shown?

	has_point (a_coordinate: REAL_COORDINATE): BOOLEAN is
			-- 
		require
			a_coordinate_exists: a_coordinate /= Void
		deferred
		end
	
feature -- Access

	standard_color: EV_COLOR is
			-- Standard color of element
		do
			if Internal_standard_color /= Void then
				Result := Internal_standard_color
			else
				Result := Default_standard_color
			end
		ensure
			Result_exists: Result /= Void
		end

	highlight_color: EV_COLOR is
			-- Highlight color of element
		do
			if Internal_highlight_color /= Void then
				Result := Internal_highlight_color
			else
				Result := Default_highlight_color
			end
		ensure
			Result_exists: Result /= Void
		end

feature -- Element change

	set_highlight_color (a_color: EV_COLOR) is
			-- Set highlight color to `a_color'.
		require
			a_color_exists: a_color /= Void
		do
			Internal_highlight_color := a_color
		ensure
			highlight_color_is_a_color: highlight_color = a_color
		end
		
	set_standard_color (a_color: EV_COLOR) is
			-- Set standard color to `a_color'.
		require
			a_color_exists: a_color /= Void
		do
			Internal_standard_color := a_color
		ensure
			standard_color_is_a_color: standard_color = a_color
		end

feature -- Basic operations

	reinitialize is
			-- Reinitialize element.
		do
			basic_representation.wipe_out
			highlighted_representation.wipe_out
			marked_representation.wipe_out
			initialize
		end
		
	show is
			-- Show element.
		do
			insert_list (basic_representation)
			shown := True
		ensure
			shown: shown
		end
		
	hide is
			-- Hide element.
		require
			shown: shown
		do
			remove_list (basic_representation)
			shown := False
		ensure
			not_shown: not shown
		end
		

	highlight is
			-- Highlight element.
		require
			shown: shown
		do
			if not highlighted then
				put_list (basic_representation, highlighted_representation)				
			end
			highlighted := True
		ensure
			highlighted: highlighted
			still_shown: shown 
		end

	unhighlight is
			-- Unhighlight element.
		require
			shown: shown
		do
			if highlighted then
				remove_list (highlighted_representation)			
			end
			highlighted := False
		ensure
			not_highlighted: not highlighted
			still_shown: shown
		end
		
	mark is
			-- Mark element.
		require
			shown: shown
		do
			put_list (basic_representation, marked_representation)
			marked := True
		ensure
			marked: marked
			still_shown: shown
		end

	unmark is
			-- Unmark element.
		require
			shown: shown
		do
			remove_list (marked_representation)
			marked := False
		ensure
			not_marked: not marked
			still_shown: shown
		end

feature {NONE} -- Implementation

	Default_standard_color: EV_COLOR is
			-- Default standard color
		deferred
		ensure
			Result_exists: Result /= Void
		end

	Default_highlight_color: EV_COLOR is
			-- Default highlight color
		deferred
		ensure
			Result_exists: Result /= Void
		end
		
	Internal_standard_color: EV_COLOR
			-- Internal standard color set by `set_standard_color'
	
	Internal_highlight_color: EV_COLOR
			-- Internal highlight color set by `set_highlight_color'
	
	basic_representation: LINKED_LIST [DRAWABLE_OBJECT]
			-- Basic graphical representation of element

	highlighted_representation: LINKED_LIST [DRAWABLE_OBJECT]
			-- Graphical representation of highlighted element

	marked_representation: LINKED_LIST [DRAWABLE_OBJECT]
			-- Graphical representation of marked element

	initialize is
			-- Initialize graphical representations.
		require
			basic_representation_empty: basic_representation.is_empty
			highlighted_representation_empty: 
					highlighted_representation.is_empty
			marked_representation_empty: marked_representation.is_empty
		deferred
		end
		
invariant

	basic_representation_exists: basic_representation /= Void
	highlighted_representation_exists: highlighted_representation /= Void
	marked_representation_exists: marked_representation /= Void		

end -- class ELEMENT_DISPLAYER

--|--------------------------------------------------------
--| This file is Copyright (C) 2004 by ETH Zurich.
--|
--| For questions, comments, additions or suggestions on
--| how to improve this package, please write to:
--|
--|     Michela Pedroni <michela.pedroni@inf.ethz.ch>
--|
--|--------------------------------------------------------
