indexing
	description: 
		"[
			Objects that given a graph calculate shortest paths.
		 ]"
		 
	author: "Michela Pedroni, ETH Zurich"
	date: "$Date: 2004/10/20 10:03:04 $"
	revision: "$Revision: 1.2 $"

deferred class SHORTEST_PATH_CALCULATOR [G -> HASHABLE, H]

feature -- Initialization

	make (a_graph: GRAPH [G, H]) is
			-- Set `graph' on which the shortest path will be calculated to `a_graph'.
		require
			a_graph_exists: a_graph /= Void
		do
			graph := a_graph
		ensure
			graph_set: graph = a_graph
		end		
		
feature -- Basic operations

	initialize, reinitialize (a_weight_function: FUNCTION [ANY, TUPLE [H], DOUBLE]; a_validity_criterium: FUNCTION [ANY, TUPLE [H], BOOLEAN]) is
			-- Initialize the shortest path calculator and set the weight function to `a_weight_function' and the validity criterium to `a_validity_criterium'.
		require
			a_weight_function_exists: a_weight_function /= Void
			a_validity_criterium_exists: a_validity_criterium /= Void
		do
			path_exists := False
			shortest_path := Void
			weight_function := a_weight_function
			validity_criterium := a_validity_criterium
		ensure
			weight_function_set: weight_function = a_weight_function
			validity_criterium_set: validity_criterium = a_validity_criterium
			path_not_exists: not path_exists 
			shortest_path_is_void: shortest_path = Void
		end

	calculate_shortest_path (a_source, a_target: G) is
			-- Calculate the shortest path between `a_source' and `a_target'.
		require
			weight_function_is_set: weight_function /= Void
			validity_criterium_is_set: validity_criterium /= Void
			a_source_exists: a_source /= Void
			a_target_exists: a_target /= Void
			a_source_in_graph: graph.has_node (a_source)
			a_target_in_graph: graph.has_node (a_target)
		deferred
		ensure
			path_exists_implies_shortest_path_set: 	
				not path_exists implies shortest_path = Void or else 
				path_exists implies (shortest_path /= Void) and shortest_path.from_node = a_source and shortest_path.to_node = a_target
		end

feature -- Status report

	path_exists: BOOLEAN
			-- Has a shortest path been found at last calculation?
			
feature -- Access

	shortest_path: SHORTEST_PATH [G, H]
			-- Outcome of last shortest path calculation (see `calculate_shortest_path')

	weight_function: FUNCTION [ANY, TUPLE [H], DOUBLE]
			-- Weight function used for shortest path calculation 
			-- (to be set with `initialize' and `reinitialize')
	
	validity_criterium: FUNCTION [ANY, TUPLE [H], BOOLEAN]
			-- Criterium for edges (if `validity_criterium' for a given edge return `False', the edge will not be admitted for the shortest path calculation)
			-- (to be set with `initialize' and `reinitialize')

	graph: GRAPH [G,H]
			-- Graph on which the shortest path calculation is operated

invariant
	
	graph_set: graph /= Void
	path_exists_implies_shortest_path_exists: path_exists implies shortest_path /= Void
	
end -- class SHORTEST_PATH_CALCULATOR

--|--------------------------------------------------------
--| This file is Copyright (C) 2004 by ETH Zurich.
--|
--| For questions, comments, additions or suggestions on
--| how to improve this package, please write to:
--|
--|     Michela Pedroni <michela.pedroni@inf.ethz.ch>
--|
--|--------------------------------------------------------
