indexing
	description: 
		"[
			Directed graphs implemented with adjacency lists (located in NODE).
			To be replaced by EiffelBase Graph library.
		 ]"
		 
	author: "Michela Pedroni, ETH Zurich"
	date: "$Date: 2004/10/18 05:09:34 $"
	revision: "$Revision: 1.2 $"

class GRAPH [G -> HASHABLE, H]

create 
	make

feature -- Initialization

	make is
			-- Initialize the graph.
		do
			create nodes.make (0)
			edges_count := 0
			create cursor.make (nodes)
		ensure
			edges_count_is_zero: edges_count = 0
			nodes_count_is_zero: nodes_count = 0
		end
		
feature -- Element change

	add_edge (a_source_key, a_target_key: G; a_data: H) is
			-- Add a new directed edge from `a_source_key' to `a_target_key' with data `a_data'.
		require
			source_exists: a_source_key /= Void
			target_exists: a_target_key /= Void
			source_available: has_node (a_source_key)
			target_available: has_node (a_target_key)
		local
			e: EDGE [G, H]
		do
			create e.make (a_source_key, a_target_key, a_data)
			nodes.item (a_source_key).add_edge (e)
			edges_count := edges_count + 1
		ensure
			one_more: edges_count = old edges_count + 1
		end
		
	add_node (a_key: G) is
			-- Add a new node with key (and data) `a_key' to the graph.
		require
			node_not_already_in_set: not has_node (a_key)
		local
			n: NODE [G, H]
		do
			create n.make (a_key)
			nodes.force (n, a_key)
		ensure
			one_more: nodes_count = old nodes_count + 1
			node_in_graph: has_node (a_key)
		end

feature -- Removal

	remove_edges (a_source_key, a_target_key: G; a_data: H) is
			-- Remove all edges that have key `a_key' and data `a_data'
		require
			a_source_key_exists: a_source_key /= Void
			a_target_key_exists: a_target_key /= Void
			a_data_exists: a_data /= Void
		local
			el1, el2: DS_LINKED_LIST [EDGE [G, H]]
		do
			el1 := edges_between (a_source_key, a_target_key)
			el2 := nodes.item (a_source_key).edges
			from
				el1.start
			until
				el1.off
			loop
				el2.start
				el2.search_forth (el1.item_for_iteration)
				if not el2.off then
					el2.remove_at				
				end
				el1.forth
			end
		end
		
feature -- Status report

	has_node (a_key: G): BOOLEAN is
			-- Is there a node with key `a_key' in this graph?
		require
			a_key_exists: a_key /= Void
		do
			Result := nodes.has (a_key)
		end
	
	has_edge (a_source_key, a_target_key: G): BOOLEAN is
			-- Is there at least one edge between node with key `a_source_key' and node with key `a_target_key'?
		require
			a_source_key_exists: a_source_key /= Void
			a_source_key_in_graph: has_node (a_source_key)
			a_target_key_exists: a_target_key /= Void
			a_target_key_in_graph: has_node (a_target_key)
		do
			from
				nodes.item (a_source_key).edges.start	
			until
				Result or else nodes.item (a_source_key).edges.off
			loop
				if nodes.item (a_source_key).edges.item_for_iteration.target_key = a_target_key then
					Result := True
				end
			end
		end

feature -- Access

	nodes_count: INTEGER is
			-- Number of nodes in the graph
		do
			Result := nodes.count
		ensure
			count_is_right: Result = nodes.count
		end
		
	edges_count: INTEGER 
			-- Number of edges in the graph

	edges_between (a_source_key, a_target_key: G): DS_LINKED_LIST [EDGE [G, H]] is
			-- Edges going from node with key `s' to node with key `a_target_key' 
			-- (recomputed at each call)
		require
			a_source_key_exists: a_source_key /= Void
			a_source_key_in_graph: has_node (a_source_key)
			a_target_key_exists: a_target_key /= Void
			a_target_key_in_graph: has_node (a_target_key)
		local
			e: DS_LINKED_LIST [EDGE [G, H]]
		do
			create Result.make
			e := nodes.item (a_source_key).edges
			from
				e.start
			until
				e.off
			loop
				if e.item_for_iteration.target_key = a_target_key then
					Result.force_last (e.item_for_iteration)
				end
				e.forth
			end
		end
	
	outgoing_edges (a_source_key: G): DS_LINKED_LIST [EDGE [G, H]] is
			-- Outgoing edges of node with key `a_source_key' 
			-- (recomputed at each call)
		require
			a_source_key_exists: a_source_key /= Void
			a_source_key_in_graph: has_node (a_source_key)
		local
			e: DS_LINKED_LIST [EDGE [G, H]]
		do
			create Result.make
			e := nodes.item (a_source_key).edges
			from
				e.start
			until
				e.off
			loop
				Result.force_last (e.item_for_iteration)
				e.forth
			end
		end

	node (a_key: G): NODE [G, H] is
			-- Node with key `a_key'
		require
			a_key_exists: a_key /= Void
			a_key_in_graph: has_node (a_key)
		do
			Result := nodes.item (a_key)
		ensure
			Result_exists: Result /= Void
			Result_is_correct: Result.data = a_key
		end
		
feature -- Iteration (traversing all nodes in no special order)
	
	start is 
			-- Start with some node
		do
			cursor.start
		end
		
	forth is
			-- Advance iterator one position
		require
			not_off: not off
		do
			cursor.forth
		end
		
	item_for_iteration: G is
			-- Current item 
		require
			not_off: not off
		do
			Result := cursor.item.data
		ensure
			Result_exists: Result /= Void
		end
		
	off: BOOLEAN is
			-- Is the cursor off?
		do
			Result := cursor.off
		end
		
feature {GRAPH_TRAVERSER} -- Implementation

	nodes: DS_HASH_TABLE [NODE [G, H], G]
			-- Hash table with all the nodes of the graph
			
	cursor: DS_HASH_TABLE_CURSOR [NODE [G, H], G]
			-- Cursor for iterating over all nodes (no specific order)
			-- Use BF_GRAPH_TRAVERSER and DF_GRAPH_TRAVERSER for tree traversal

end -- class GRAPH

--|--------------------------------------------------------
--| This file is Copyright (C) 2004 by ETH Zurich.
--|
--| For questions, comments, additions or suggestions on
--| how to improve this package, please write to:
--|
--|     Michela Pedroni <michela.pedroni@inf.ethz.ch>
--|
--|--------------------------------------------------------
