indexing
	description: 
		"[
			Directed edges in a graph connecting two nodes.
			G: type of data stored in nodes
			H: type of data stored in edge (e.g. for shortest path calculation)
		 ]"

	author: "Michela Pedroni, ETH Zurich"
	date: "$Date: 2004/10/20 10:03:04 $"
	revision: "$Revision: 1.2 $"

class EDGE [G -> HASHABLE, H]
	
create 
	make
	
feature -- Initialization

	make (a_source_key, a_target_key: G; a_data: H) is
			-- Make a directed edge with data `a_data' going from node with key `a_source_key' to node with key `a_target_key'.
		require
			a_source_key_exists: a_source_key /= Void
			a_target_key_exists: a_target_key /= Void
			source_key_not_set: source_key = Void
			target_key_not_set: target_key = Void
		do
			source_key := a_source_key
			target_key := a_target_key
			data := a_data
		ensure
			source_key_set: source_key = a_source_key
			target_key_set: target_key = a_target_key
			data_set: data = a_data
		end
		
feature -- Access

	source_key: G
			-- Key of start node (`Current' goes from node of `source_key' to node of `target_key')
	
	target_key: G
			-- Key of end node (`Current' goes from node of `source_key' to node of `target_key')
	
	data: H
			-- Data stored in this edge (e.g. a weight)

	other_end (a_key: G): G is
			-- Key of the node on the other end (relativ to `a_key') of the edge
		require
			a_key_exists: a_key /= Void
			has_a_key: has (a_key)
		do
			if a_key = source_key then
				Result := target_key
			else
				Result := source_key
			end
		ensure
			has_other_end: has (Result)
		end

feature -- Status report

	has (a_key: G): BOOLEAN is
			-- Is `Current' going from or to node with key `a_key'?
		require
			a_key_exists: a_key /= Void
		do
			Result := (a_key = source_key) or (a_key = target_key)
		end
		
feature -- Element change

	set_data (a_data: H) is
			-- Set `data' to `a_data'.
		require
			a_data_exists: a_data /= Void
		do
			data := a_data
		ensure 
			data_set: data = a_data
		end
		
invariant
	
	source_key_exists: source_key /= Void
	target_key_exists: target_key /= Void
	
end -- class EDGE

--|--------------------------------------------------------
--| This file is Copyright (C) 2004 by ETH Zurich.
--|
--| For questions, comments, additions or suggestions on
--| how to improve this package, please write to:
--|
--|     Michela Pedroni <michela.pedroni@inf.ethz.ch>
--|
--|--------------------------------------------------------
