indexing
	description: 
		"[
			Shortest path calculator using the classic single-source dijkstra algorithm.
		 ]"
		 
	author: "Michela Pedroni, ETH Zurich"
	date: "$Date: 2004/10/20 10:03:04 $"
	revision: "$Revision: 1.2 $"

class DIJKSTRA_SHORTEST_PATH_CALCULATOR [G -> HASHABLE, H] inherit 

	SHORTEST_PATH_CALCULATOR [G, H]
		redefine 
			make 
		end
	
create 
	make

feature -- Initialization
	
	make (a_graph: GRAPH [G, H]) is
			-- Set `graph' on which the shortest path will be calculated to `a_graph'.
		do
			Precursor {SHORTEST_PATH_CALCULATOR} (a_graph)
			create settled_nodes.make (a_graph.nodes_count)
			create unsettled_nodes.make (a_graph.nodes_count)
			create shortest_distances.make (a_graph.nodes_count)
			create shortest_path_tree.make (a_graph.nodes_count)
		end

feature -- Basic operations

	calculate_shortest_path (a_source_key, a_target_key: G) is
			-- Calculate the shortest path from `a_source_key' to `a_target_key'. Result is stored in `shortest_path'.
		local
			min_item: G
			e: EDGE [G, H]
			g: G
		do
			reset (a_source_key)
			unsettled_nodes.force (a_source_key)
			shortest_distances.force (0.0, a_source_key)
			
			from
				
			until
				unsettled_nodes.count = 0
			loop
				min_item := extract_minimum
				unsettled_nodes.remove (min_item)
				settled_nodes.force (min_item)
				relax_neighbors (min_item)
			end
			create shortest_path.make
			if shortest_path_tree.has (a_target_key) then
				from
					e := shortest_path_tree.item (a_target_key)
					g := a_target_key
				until
					shortest_path = Void or else e.other_end (g) = a_source_key
				loop
					shortest_path.put_edge_front (e)
					if e.has (g) and shortest_path_tree.has (e.other_end (g)) then
						g := e.other_end (g)
						e := shortest_path_tree.item (g)
					else
						shortest_path := Void
					end
				end
				if shortest_path/= Void then
					shortest_path.put_edge_front (e)					
				end
			end			
		end

feature {NONE} -- Implementation

	reset (a_source_key: G) is
			-- Reset the data for the dijkstra single-source shortest path calculation with source `src'.
		require
			a_source_key_exists: a_source_key /= Void
			a_source_key_in_graph: graph.has_node (a_source_key)
		local
			bfs: BF_GRAPH_TRAVERSER [G, H]
		do
			settled_nodes.wipe_out
			unsettled_nodes.wipe_out
			shortest_distances.wipe_out
			shortest_path_tree.wipe_out
			create bfs.make (graph)
			from
				bfs.start_with (a_source_key)
			until
				bfs.after
			loop
				shortest_distances.force (200000000.0, bfs.item.data)
				bfs.forth
			end
		ensure
			settled_nodes.count = 0
			unsettled_nodes.count = 0
			shortest_path_tree.count = 0
		end

	extract_minimum: G is
			-- Find the node with the smallest distance
		local
			min: DOUBLE
		do
			from
				unsettled_nodes.start
				min := 2000000
			until
				unsettled_nodes.off
			loop
				if min > shortest_distances.item (unsettled_nodes.item_for_iteration) then
					Result := unsettled_nodes.item_for_iteration	
				end
				unsettled_nodes.forth
			end
		ensure
			minimum_exists: Result /= Void
		end
	
	relax_neighbors (a_key: G) is
			-- `weight' defines the weight function to be used to calculate the link's weight and criteria is used as a ko criteria to remove unwanted links
		require
			a_key_exists: a_key /= Void
			a_key_in_graph: graph.has_node (a_key)
		local
			l: DS_LINKED_LIST [EDGE [G, H]]
			e: EDGE [G, H]
		do
			from
				l := graph.outgoing_edges (a_key)
				l.start
			until
				l.off
			loop
				e := l.item_for_iteration
				if validity_criterium.item ([e.data]) and then shortest_distances.item (e.target_key) > weight_function.item ([e.data]) + shortest_distances.item (a_key) then
					shortest_distances.replace (weight_function.item ([e.data]) + shortest_distances.item (a_key), e.target_key)
					if not shortest_path_tree.has (e.target_key) then
						shortest_path_tree.force (e, e.target_key)
					else
						shortest_path_tree.replace (e, e.target_key)
					end
					unsettled_nodes.put (e.target_key)
				end
				l.forth
			end
		end
		
	settled_nodes: DS_HASH_SET [G]
			-- Nodes that are already settled (meaning the shortest path to them is final)
	
	unsettled_nodes: DS_HASH_SET [G]
			-- Nodes that are not yet settled
			
	shortest_distances: DS_HASH_TABLE [DOUBLE, G]
			-- Stores for each node the current shortest distance
	
	shortest_path_tree: DS_HASH_TABLE [EDGE [G, H], G]
			-- Stores for each node what edge has to be taken to reach it

end -- class DIJKSTRA_SHORTEST_PATH_CALCULATOR

--|--------------------------------------------------------
--| This file is Copyright (C) 2004 by ETH Zurich.
--|
--| For questions, comments, additions or suggestions on
--| how to improve this package, please write to:
--|
--|     Michela Pedroni <michela.pedroni@inf.ethz.ch>
--|
--|--------------------------------------------------------
