indexing
	description: 
		"[
			One-way transportation lines (used to build lines that go into both directions).
		 ]"
		 
	author: "Michela Pedroni, ETH Zurich"
	date: "$Date: 2004/10/20 10:03:03 $"
	revision: "$Revision: 1.4 $"

class ONE_WAY_LINE

create
	make
	
feature -- Initialization

	make is
			-- Initialize this one-way line.
		do
			create edges.make
			create internal_cursor.make (edges)
		end

feature -- Access

	first_link: LINK is
			-- First link in line
		require
			not_empty: not is_empty
		do
			Result := edges.first.data
		end
		
	first_place: PLACE is
			-- First place in line
		require
			not_empty: not is_empty
		do
			Result := edges.first.source_key
		end
		
	link, i_th_link (an_index: INTEGER): LINK is
			-- Link at index `an_index'
			-- (Performance: O(i).)
		require
			valid_index: 1 <= an_index and an_index <= count
		do
			Result := edges.item (an_index).data
		end

	place, i_th_place (an_index: INTEGER): PLACE is
			-- Place at index `an_index'.
		require
			not_empty: not is_empty
			valid_index: 1 <= an_index and an_index <= count + 1
		do
			if an_index = count + 1 then
				Result := edges.last.target_key
			else
				Result := edges.item (an_index).source_key
			end
		end
	
	link_for_iteration: LINK is
			-- Link at internal cursor position
		require
			not_off: not off
		do
			Result := internal_cursor.item.data
		end

	place_for_iteration: PLACE is
			-- Place at internal cursor position
			-- Note that last place must be accessed through `last_place' or `place (count)'
		require
			not_off: not off
		do
			Result := internal_cursor.item.source_key
		end

	last_link: LINK is
			-- Last link in line
		require
			not_empty: not is_empty
		do
			Result := edges.last.data
		end

	last_place: PLACE is
			-- Last place in line
		require
			not_empty: not is_empty
		do
			Result := edges.last.target_key
		end		
		
feature -- Cursor movement

	start is
			-- Move `a_cursor' to first position.
			-- (Performance: O(1).)
		do
			internal_cursor.start
		ensure
			empty_behavior: edges.is_empty implies after
			first_or_after: is_first xor after
		end
		
	forth is
			-- Move cursor to next position.
		require
			not_after: not after
		do
			internal_cursor.forth
		end	
		
	back is
			-- Move cursor to previous position.
		require
			not_before: not before
		do
			internal_cursor.back
		end		

	search_link_forth (a_link: LINK) is
			-- Move internal cursor to first position at or before current
			-- position where `link_for_iteration' and `a_link' are equal.
			-- Move `after' if not found.
		require
			not_off: not off or before
		local
			found: BOOLEAN
		do
			from
			until
				off or found
			loop
				if a_link = link_for_iteration then
					found := True
				else
					forth
				end
			end
		end
		
feature -- Status report

	is_first: BOOLEAN is
			-- Is cursor on first item?
		do
			Result := internal_cursor.is_first
		ensure
			not_empty: Result implies not edges.is_empty
			not_off: Result implies not off
		end

	after: BOOLEAN is
			-- Is there no valid position to right of cursor?
		do
			Result := internal_cursor.after
		end
		
	before: BOOLEAN is
			-- Is there no valid position to left of cursor?
		do
			Result := internal_cursor.before
		end
		
	off: BOOLEAN is
			-- Is there no item at cursor position?
		do
			Result := internal_cursor.off
		end
		
	is_empty: BOOLEAN is
			-- Is container empty?
		do
			Result := edges.is_empty
		end		
		
feature -- Measurement

	count: INTEGER is
			-- Number of links that are part of this one-way line
		do
			Result := edges.count
		end

feature -- Status report

	has_link (a_link: LINK): BOOLEAN is
			-- Is `a_link' part of the one-way line?
		require
			a_link_exists: a_link /= Void
		do
			from
				start
			until
				off or Result
			loop
				if link_for_iteration = a_link then
					Result := True
				end
				forth
			end
		end	
		
	has_place (a_place: PLACE): BOOLEAN is
			-- Is `a_place' part of the one-way line?
		require
			a_place_exists: a_place /= Void
		do
			from
				start
			until
				off or Result
			loop
				if place_for_iteration = a_place then
					Result := True
				end
				forth
			end
			if last_place = a_place then
				Result := True
			end
		end
		
feature -- Element change

	put_end (a_link: LINK) is
			-- Put link `a_link' at the end of the line.
		require
			not_empty: not is_empty
			a_link_not_in_line: not has_link (a_link)
			a_link_fits_directed: count > 0 and then a_link.directed implies (a_link.from_place = last_place and not has_place (a_link.to_place))
			a_link_fits_undirected: count > 0 and then not a_link.directed implies (a_link.from_place = last_place and not has_place (a_link.to_place)) 
																				or (a_link.to_place = last_place and not has_place (a_link.from_place))
		do
			if count < 1 or else a_link.from_place = last_place then
				edges.put_last (create {EDGE [PLACE, LINK]}.make (a_link.from_place, a_link.to_place, a_link))
			else
				edges.put_last (create {EDGE [PLACE, LINK]}.make (a_link.to_place, a_link.from_place, a_link))
			end
		ensure
			one_more: count = old count + 1
		end
		
	put_front (a_link: LINK) is
			-- Put link `a_link' at the front of the line.
		require
			a_link_not_in_line: not has_link (a_link)
			a_link_fits_directed: count > 0 and then a_link.directed implies (a_link.to_place = first_place and not has_place (a_link.from_place))
			a_link_fits_undirected: count > 0 and then not a_link.directed implies (a_link.to_place = first_place and not has_place (a_link.from_place))
																				or (a_link.from_place = first_place and not has_place (a_link.to_place))
		do
			if count < 1 or else a_link.to_place = first_place then
				edges.put_first (create {EDGE [PLACE, LINK]}.make (a_link.from_place, a_link.to_place, a_link))
			else
				edges.put_first (create {EDGE [PLACE, LINK]}.make (a_link.to_place, a_link.from_place, a_link))
			end
		ensure
			one_more: count = old count + 1
		end		
		
	put_first (a_link: LINK; reversed: BOOLEAN) is
			-- Insert `a_link' as the first link into the one-way line.
			-- `reversed' denotes whether the link should be reversed (starting place of line would then be the `to_place' of `a_link').
		require
			is_empty: count = 0
			a_link_exists: a_link /= Void
			a_link_not_in_line: not has_link (a_link)
		do
			if reversed then
				edges.put_first (create {EDGE [PLACE, LINK]}.make (a_link.to_place, a_link.from_place, a_link))
			else
				edges.put_first (create {EDGE [PLACE, LINK]}.make (a_link.from_place, a_link.to_place, a_link))
			end
		ensure
			one_more: count = 1
		end

feature -- Removal

	prune_first_link is
			-- Remove first link from one-way line.
			-- Move the internal cursor `off'.
		require
			not_empty: not is_empty
		do
			edges.prune_first (1)
			internal_cursor.go_after
		ensure
			one_less: count = old count - 1
			first_link_gone: is_empty or else first_link /= old first_link
		end
		
	prune_last_link is
			-- Remove last link from one-way line.
			-- Move the internal cursor `off'.
		require
			not_empty: not is_empty
		do
			edges.prune_last (1)
			internal_cursor.go_after
		ensure
			one_less: count = old count - 1
			last_link_gone: is_empty or else last_link /= old last_link
		end
		
		
feature {NONE} -- Implementation

	edges:  DS_LINKED_LIST [EDGE [PLACE, LINK]]
			-- List of all (directed) edges in the one-way line
			
	internal_cursor: DS_LINKED_LIST_CURSOR [EDGE [PLACE, LINK]]
			-- Cursor used for traversing the one-way line
	
end -- class ONE_WAY_LINE

--|--------------------------------------------------------
--| This file is Copyright (C) 2004 by ETH Zurich.
--|
--| For questions, comments, additions or suggestions on
--| how to improve this package, please write to:
--|
--|     Michela Pedroni <michela.pedroni@inf.ethz.ch>
--|
--|--------------------------------------------------------
