indexing
	description: 
		"[
			Shared registry of known XML node processors.
		 ]"

	status:	"See notice at end of class"
	author: "Patrick Schoenbach, Michela Pedroni"
	date: "$Date: 2004/10/20 10:03:03 $"
	revision: "$Revision: 1.3 $"

class NODE_PROCESSOR_REGISTRY

feature {NONE} -- Access

	processor (a_name: STRING): NODE_PROCESSOR is
			-- Node processor named `a_name'
		require
			name_exists: a_name /= Void
			name_not_empty: not a_name.is_empty
			element_exists: has_processor (a_name)
		do
			Result := clone (Processor_registry @ a_name)
			Result.make
		ensure
			Result_exists: Result /= Void
		end

feature -- Status setting

	city_factory: CITY_FACTORY
			-- Reference to the city factory
			
	city: CITY is
			-- City that is built
		do
			Result := city_factory.city
		end
		
	set_city_factory (a_factory: CITY_FACTORY) is
			-- Set city reference.
		do
			city_factory := a_factory
		end

	processor_registered (a_processor: NODE_PROCESSOR): BOOLEAN is
			-- Is `a_processor' registered?
		do
			Result := Processor_registry.has_item (a_processor)
		end
		
feature -- Basic operations

	register_processor (a_processor: NODE_PROCESSOR) is
			-- Register `a_processor' in registry.
		require
			processor_exists: a_processor /= Void
			not_registered: not processor_registered (a_processor)
		do
			Processor_registry.force (a_processor, a_processor.name)
		ensure
			registered: Processor_registry.has_item (a_processor)
		end

	register_allowed_subnode (a_parent_name: STRING; a_subnode_name: STRING) is
			-- Register processor with name `a_subnode_name' as allowed subnode of processor with name `a_parent_name'.
			-- Use `register_processor' for new node types and 
			-- `register_allowed_subnode' for registering their subnode types.
		local
			a: ARRAY [STRING]
		do
			if Allowed_subnode_registry.has (a_parent_name) then
				Allowed_subnode_registry.item (a_parent_name).force (a_subnode_name, Allowed_subnode_registry.item (a_parent_name).count + 1)
			else
				a := << a_subnode_name >>
				a.compare_objects
				Allowed_subnode_registry.force ( a, a_parent_name)
			end
		end
			
feature {NONE} -- Status report

	has_processor (a_name: STRING): BOOLEAN is
			-- Is processor named `a_name' available?
		require
			name_exists: a_name /= Void
			name_not_empty: not a_name.is_empty
		do
			Result := Processor_registry.has (a_name)
		end
		
feature {NONE} -- Constants

	Default_registry_capacity: INTEGER is 100
			-- Default capacity of registry
		
feature {NONE} -- Implementation

	Processor_registry: DS_HASH_TABLE [NODE_PROCESSOR, STRING] is
			-- System-wide registry for known map node processors
			-- Use `register_processor' for new node types and 
			-- `register_allowed_subnode' for registering their subnode types.
		once
			create Result.make (Default_registry_capacity)
			Result.put (create {CITY_NODE_PROCESSOR}.make, "city")
			Result.put (create {PLACES_NODE_PROCESSOR}.make, "places")
			Result.put (create {PLACE_NODE_PROCESSOR}.make, "place")
			Result.put (create {LINES_NODE_PROCESSOR}.make, "lines")
			Result.put (create {LINE_NODE_PROCESSOR}.make, "line")
			Result.put (create {POINT_NODE_PROCESSOR}.make, "point")
			Result.put (create {STATION_NODE_PROCESSOR}.make, "station")			
			Result.put (create {LINK_NODE_PROCESSOR}.make, "link")	
			Result.put (create {DESCRIPTION_NODE_PROCESSOR}.make, "description")
		ensure
			Result_exists: Result /= Void
			Result_not_empty: not Result.is_empty
		end

	Allowed_subnode_registry: DS_HASH_TABLE [ARRAY [STRING], STRING] is
			-- System-wide registry for known map node processors
			-- Use `register_processor' for new node types and 
			-- `register_allowed_subnode' for registering their subnode types.
		local
			a: ARRAY [STRING]
		once
			create Result.make (Default_registry_capacity)
			a := << "places", "lines" >>
			a.compare_objects
			Result.put (a, "city")
			a := << "place", "station" >>
			a.compare_objects
			Result.put (a, "places")
			a := << "point", "description" >>
			a.compare_objects
			Result.put (a, "place")
			a := <<  >>
			a.compare_objects
			Result.put (a, "point")
			a := << "link" >>
			a.compare_objects
			Result.put (a, "line")
			a := << "line" >>
			a.compare_objects
			Result.put (a, "lines")
			a := << "point" >>
			a.compare_objects
			Result.put (a, "link")
		end
	
		
end

--|--------------------------------------------------------
--| This file is Copyright (C) 2003 by ETH Zurich.
--|
--| For questions, comments, additions or suggestions on
--| how to improve this package, please write to:
--|
--|     Patrick Schoenbach <pschoenb@gmx.de>
--|		Michela Pedroni <pedronim@inf.ethz.ch>
--|
--|--------------------------------------------------------
