indexing
	description: 
		"[
			Factory for city elements like places, links, and the city itself.
		 ]"
		 
	author: "Michela Pedroni, ETH Zurich"
	date: "$Date: 2004/10/20 10:03:03 $"
	revision: "$Revision: 1.4 $"

class CITY_FACTORY inherit 

	LINE_LINK_TYPE_CONSTANTS
	
create
	make

feature -- Initialization

	make, reset_factory is
			-- Reset the city factory.
		do
			internal_city := Void
			internal_place := Void
			internal_link := Void
			internal_line := Void
		ensure
			everything_void: internal_city = Void and internal_place = Void and internal_link = Void and internal_line = Void
		end

feature -- Element change

	set_displayer_factory (a_factory: DISPLAYER_FACTORY) is
			-- Set `displayer_factory' to `a_factory'.
		do
			displayer_factory := a_factory
		ensure
			displayer_factory_set: displayer_factory = a_factory
		end
		
feature -- City building

	build_city (a_name: STRING) is
			-- Generate a new city object with name `a_name'.
			-- (Access the generated object through feature `city')
		require
			a_name_exists: a_name /= Void
			a_name_not_empty: not a_name.is_empty
		do
			create internal_city.make (a_name)
			if displayer_factory /= Void then
				displayer_factory.build_city_displayer (internal_city)
				internal_city.set_displayer (displayer_factory.new_city_displayer)
			end
		ensure
			city_created: city /= Void
			city_has_name: equal (city.name, a_name)
		end

	city: CITY is
			-- Generated city object
		require
			city_available: has_city
		do
			Result := internal_city
		ensure
			Result_exists: Result /= Void
		end
	
	has_city: BOOLEAN is
			-- Is there a city object available?
		do
			Result := internal_city /= Void
		end
		
feature -- Place building

	build_place (a_name: STRING; a_city: CITY) is
			-- Generate a new place object with name `a_name' belonging to city `a_city'.
			-- (Access the generated object through feature `place')
		require
			a_name_exists: a_name /= Void
			a_name_not_empty: not a_name.is_empty
			city_exists: has_city
			unique_name: not city.has_place (a_name)
		do
			create internal_place.make (a_name)
			a_city.places.force (internal_place, internal_place.name)			
			a_city.transport_network.add_place (internal_place)
			internal_place.set_city (a_city)
			if displayer_factory /= Void then
				displayer_factory.build_place_displayer (internal_place)
				internal_place.set_displayer (displayer_factory.new_place_displayer)
			end
		ensure
			place_created: place /= Void
			place_has_name: equal (place.name, a_name)
			place_in_city: a_city.has_place (a_name)
			place_has_city: place.city = a_city
		end
		
	place: PLACE is
			-- Generated place object
		require
			place_available: has_place
		do
			Result := internal_place
		ensure
			Result_exists: Result /= Void
		end

	has_place: BOOLEAN is
			-- Is there a place object available?
		do
			Result := internal_place /= Void
		end

feature -- Station building

	build_station (a_name: STRING; a_city: CITY) is
			-- Generate a new station object with name `name'.
			-- (Access the generated object through feature `station')
		require
			a_name_exists: a_name /= Void
			a_name_not_empty: not a_name.is_empty
			city_exists: has_city
			unique_name: not city.has_place (a_name)
		do
			create internal_station.make (a_name)
			a_city.places.force (internal_station, internal_station.name)			
			a_city.transport_network.add_place (internal_station)
			internal_station.set_city (a_city)
			if displayer_factory /= Void then
				displayer_factory.build_station_displayer (internal_station)
				internal_station.set_displayer (displayer_factory.new_station_displayer)
			end
		ensure
			station_created: station /= Void
			station_has_name: equal (station.name, a_name)
			station_in_city: a_city.has_place (a_name)
			station_has_city: a_city = station.city
		end
		
	station: STATION is
			-- Generated station object
		require
			station_available: has_station
		do
			Result := internal_station
		ensure
			Result_exists: Result /= Void
		end

	has_station: BOOLEAN is
			-- Is there a station object available?
		do
			Result := internal_station /= Void
		end

feature -- Link building

	build_link (a_place1, a_place2: STRING; directed: BOOLEAN; a_city: CITY; a_line: LINE) is
			-- Generate a new link object going from place with name `a_place1' to place with name `a_place2'.
			-- The generated link will belong to the last generated line and will be of the same type.
			-- (Access the generated object through feature `link')
		require
			a_city_exists: a_city /= Void
			a_place1_exists: a_city.has_place (a_place1)
			a_place2_exists: a_city.has_place (a_place2)
			a_line_exists: a_line /= Void
		do
			create internal_link.make (a_line.type, directed, a_city.places.item (a_place1), a_city.places.item (a_place2))
			a_city.links.put_last (internal_link)			
			a_city.transport_network.add_link (a_city.places.item (a_place1), a_city.places.item (a_place2), internal_link)
			if not directed then
				a_city.transport_network.add_link (a_city.places.item (a_place2), a_city.places.item (a_place1), internal_link)
			end
			a_line.add_link (internal_link)
			internal_link.set_line (a_line)
			internal_link.set_city (a_city)
			if displayer_factory /= Void then
				displayer_factory.build_link_displayer (internal_link)
				internal_link.set_displayer (displayer_factory.new_link_displayer)
			end
		ensure
			link_created: link /= Void
			link_has_line: link.line = a_line
			link_has_type: equal (link.type, a_line.type)
			link_has_source: link.from_place = a_city.places.item (a_place1)
			link_has_target: link.to_place = a_city.places.item (a_place2)
			link_has_city: link.city = a_city
		end

	link: LINK is
			-- Generated link object
		require
			link_available: has_link
		do
			Result := internal_link
		ensure
			Result_exists: Result /= Void
		end

	has_link: BOOLEAN is
			-- Is there a link object available?
		do
			Result := internal_link /= Void
		end
		
feature -- Line building

	build_line (a_name: STRING; a_type: STRING; a_city: CITY) is
			-- Generate a new line object with name `name' and type `type'.
			-- (Access the generated object through feature `line')
		require
			a_name_exists: a_name /= Void
			a_name_not_empty: not a_name.is_empty
			type_is_valid: is_valid_type (a_type)
			city_exists: has_city
		do
			create internal_line.make (a_name, a_type)
			a_city.lines.force (internal_line, internal_line.name)
			internal_line.set_city (a_city)
			if displayer_factory /= Void then
				displayer_factory.build_line_displayer (internal_line)
				internal_line.set_displayer (displayer_factory.new_line_displayer)
			end
		ensure
			line_created: line /= Void
			line_has_name: equal (line.name, a_name)
			line_has_type: equal (line.type, a_type)
			line_has_city: a_city = line.city
		end

	line: LINE is
			-- Generated line object
		require
			place_available: has_line
		do
			Result := internal_line
		ensure
			Result_exists: Result /= Void
		end

	has_line: BOOLEAN is
			-- Is there a line object available?
		do
			Result := internal_line /= Void
		end

feature -- Attributes

	displayer_factory: DISPLAYER_FACTORY
			-- Factory that creates displayers for the city elements
	
feature {NONE} -- Implementation

	internal_place: PLACE
			-- Internal representation of last created place
	
	internal_station: STATION
			-- Internal representation of last created station
	
	internal_link: LINK
			-- Internal representation of last created link
	
	internal_line: LINE
			-- Internal representation of last created line
	
	internal_city: CITY
			-- Internal representation of last created city
	
end -- class CITY_FACTORY

--|--------------------------------------------------------
--| This file is Copyright (C) 2004 by ETH Zurich.
--|
--| For questions, comments, additions or suggestions on
--| how to improve this package, please write to:
--|
--|     Michela Pedroni <michela.pedroni@inf.ethz.ch>
--|
--|--------------------------------------------------------
