indexing
	description: 
		"[
			Main window of the application. Is controlled by a MAIN_CONTROLLER object.
		 ]"
		 
	author: "Michela Pedroni, ETH Zurich"
	date: "$Date: 2004/10/20 10:03:03 $"
	revision: "$Revision: 1.5 $"

class MAIN_WINDOW inherit
	
	EV_TITLED_WINDOW
		redefine
			initialize, 
			is_in_default_state
		end

	MAIN_WINDOW_CONSTANTS
		export
			{NONE} all
		undefine
			default_create, copy
		end

	KL_SHARED_FILE_SYSTEM
		undefine
			default_create, copy
		end
		
create
	default_create

feature {NONE} -- Window initialization

	initialize is
			-- Build the interface for this window.
		do
			Precursor {EV_TITLED_WINDOW}
			build_menu_bar
			build_main_container
			build_cursors
			extend (main_container)
			set_title (Window_title)
			set_size (Window_width, Window_height)
			selection_button.toggle
			set_selection_mode
		end
		
feature -- Basic operations

	set_tool_bar_sensitivity (sensitive: BOOLEAN) is
			-- Set sensitivity of all tool bar buttons according to `sensitive'.
		require
			tool_bar_exists: tool_bar /= Void
		local
			b: EV_BUTTON
			i: INTEGER
			pic: EV_PIXMAP
		do
			from
				tool_bar.start
				i := 1
			until
				tool_bar.after
			loop
				b ?= tool_bar.item
				if b /= Void then
					create pic
					if sensitive then
						b.enable_sensitive
						pic.set_with_named_file (File_system.pathname (Image_directory, (Button_images @ i) + Image_extension))
					else
						b.disable_sensitive
						pic.set_with_named_file (File_system.pathname (Image_directory, (Button_images @ i + Inactivity_suffix + Image_extension)))
					end
					b.set_pixmap (pic)
					i := i + 1
				end
				tool_bar.forth
			end
		end
		
	set_route_calculation_sensitivity (sensitive: BOOLEAN) is
			-- Set sensitivity of route calculation tool bar button according to `sensitive'.
		local
			pic: EV_PIXMAP
		do
			create pic
			if sensitive then
				route_calculation_button.enable_sensitive
				pic.set_with_named_file (File_system.pathname (Image_directory, (route_calculation_button_image) + Image_extension))
			else
				route_calculation_button.disable_sensitive
				pic.set_with_named_file (File_system.pathname (Image_directory, (route_calculation_button_image) + Inactivity_suffix + Image_extension))					
			end
			route_calculation_button.set_pixmap (pic)	
		end
		
feature -- Window elements

	file_menu: EV_MENU
			-- "File" menu for this window (contains New, Open, Close, Exit...)
			
	file_open_item: EV_MENU_ITEM 
			-- Menu item that reads the xml file input
		
	file_close_item: EV_MENU_ITEM
			-- Menu item that closes the application
		
	help_about_item: EV_MENU_ITEM
			-- Menu item that opens the About dialog
			
	help_menu: EV_MENU
			-- "Help" menu for this window (contains About...)
			
	zoom_in_button: EV_BUTTON
			-- Zoom in button in tool bar
	
	zoom_out_button: EV_BUTTON
			-- Zoom out button in tool bar
	
	selection_button: EV_TOGGLE_BUTTON
			-- Mark a stop
	
	move_button: EV_TOGGLE_BUTTON
			-- Move map with left mouse button
			
	information_button: EV_TOGGLE_BUTTON
			-- Information button -> add stops to lines

	name_display_button: EV_TOGGLE_BUTTON
			-- Toggle name display button
			
	route_calculation_button: EV_BUTTON
			-- Route calculation button in tool bar
	
	tool_bar: EV_VERTICAL_BOX
			-- Tool bar
	
	canvas: TRAFFIC_CANVAS
			-- The Canvas Widget
			
	main_container: EV_VERTICAL_BOX
			-- Main container (contains all widgets displayed in this window)

	viewport: EV_VIEWPORT
			-- To display the canvas

	move_cursor: EV_CURSOR
			-- Cursor for move mode

	information_cursor: EV_CURSOR
			-- Cursor for information mode

	traffic_console: TRAFFIC_CONSOLE
			-- Console for textual output
	
	button_panel: BUTTON_PANEL
			-- Button panel for example and exercise buttons
			
feature {NONE} -- Window setup

	build_main_container is
			-- Create and populate `main_container'.
		require
			main_container_not_yet_created: main_container = Void
		local
			vb: EV_VERTICAL_SPLIT_AREA
			hb1, hb2: EV_HORIZONTAL_BOX
			fr: EV_FRAME
		do
			create viewport
			create vb
			create hb1
			create hb2
			create fr
			create canvas.make
			create main_container
			create traffic_console
			create button_panel

			build_button_bar
			
			viewport.set_minimum_height (400)
			viewport.set_minimum_width (600)
			viewport.extend (canvas)
			fr.extend (viewport)
			button_panel.set_up

			traffic_console.disable_edit
			canvas.set_zoom_limits (0.5, 10.0)

			hb1.extend (tool_bar)
			hb1.extend (fr)
			hb1.disable_item_expand (tool_bar)

			vb.extend (hb1)
			vb.extend (traffic_console)
			hb2.extend (vb)
			hb2.extend (button_panel)
			hb2.disable_item_expand (button_panel)
			main_container.extend (hb2)
			main_container.set_padding (10)


		ensure
			main_container_created: main_container /= Void
		end
		
	build_menu_bar is
			-- Create and populate `standard_menu_bar'.
		require
			no_menu_bar: menu_bar = Void
		local
			m_bar: EV_MENU_BAR
		do
				-- Create the menu bar.
			create m_bar

				-- Add the "File" menu.
			create file_menu.make_with_text (File_menu_text)
			create file_open_item.make_with_text (File_open_item_text)
			create file_close_item.make_with_text (File_exit_item_text)
			file_menu.extend (file_open_item)
			file_menu.extend (create {EV_MENU_SEPARATOR})
			file_menu.extend (file_close_item)
			m_bar.extend (file_menu)

				-- Add the "Help" menu.
			create help_menu.make_with_text (Help_menu_text)
			create help_about_item.make_with_text (Help_about_item_text)
			help_menu.extend (help_about_item)
			m_bar.extend (help_menu)
			
				-- Set the menu bar for this window
			set_menu_bar (m_bar)
		ensure
			menu_bar_exists: menu_bar /= Void
			menu_bar_not_empty: not menu_bar.is_empty
		end

	build_button_bar is
			-- Create and populate `button_bar'.
		local
			sep: EV_VERTICAL_SEPARATOR
		do
			create tool_bar
			tool_bar.set_padding (Button_padding)

			create zoom_in_button.make_with_text_and_action ("", agent Canvas.zoom_in (Zoom_factor_stepwise))
			zoom_in_button.set_tooltip (Zoom_in_tooltip)
			tool_bar.extend (zoom_in_button)
			tool_bar.disable_item_expand (zoom_in_button)

			create zoom_out_button.make_with_text_and_action ("", agent Canvas.zoom_out (Zoom_factor_stepwise))
			zoom_out_button.set_tooltip (Zoom_out_tooltip)
			tool_bar.extend (zoom_out_button)
			tool_bar.disable_item_expand (zoom_out_button)

			create sep
			tool_bar.extend (sep)
			tool_bar.disable_item_expand (sep)
			
			create selection_button
			selection_button.set_tooltip (Selection_tooltip)
			selection_button.select_actions.extend (agent set_selection_mode) -- toggle_procedure (selection_button))
			tool_bar.extend (selection_button)
			tool_bar.disable_item_expand (selection_button)

 			create move_button
			move_button.set_tooltip (Move_tooltip)
			move_button.select_actions.extend (agent set_move_mode)
			tool_bar.extend (move_button)
			tool_bar.disable_item_expand (move_button)

			create information_button
			information_button.set_tooltip (Info_tooltip)
			information_button.select_actions.extend 
				(agent set_information_mode)
			tool_bar.extend (information_button)
			tool_bar.disable_item_expand (information_button)

			create sep
			tool_bar.extend (sep)
			tool_bar.disable_item_expand (sep)

			create name_display_button
			name_display_button.set_tooltip (Name_display_tooltip)
			tool_bar.extend (name_display_button)
			tool_bar.disable_item_expand (name_display_button)

			create sep
			tool_bar.extend (sep)
			tool_bar.disable_item_expand (sep)
			
			create route_calculation_button
			route_calculation_button.set_tooltip (Route_calculation_tooltip)
			tool_bar.extend (route_calculation_button)
			tool_bar.disable_item_expand (route_calculation_button)

			tool_bar.set_border_width (5)

			set_tool_bar_sensitivity (False)
		end

	build_cursors is
			-- Build cursors.
		require
			no_move_cursor: move_cursor = Void
			no_information_cursor: information_cursor = Void
		local
			pic: EV_PIXMAP
		do
			create pic
			pic.set_with_named_file (File_system.pathname (Image_directory, Move_cursor_shape + Image_extension))
			create move_cursor.make_with_pixmap 
				(pic, Cursor_hotspot_x, Cursor_hotspot_y)

			create pic
			pic.set_with_named_file (File_system.pathname (Image_directory, Info_cursor_shape + Image_extension))
			create information_cursor.make_with_pixmap 
				(pic, 0, 0)
		ensure
			move_cursor_exists: move_cursor /= Void
			information_cursor_exists: information_cursor /= Void
		end
		
feature {NONE} -- Implementation

	set_selection_mode is
			-- Set element selection mode.
		do
			mode := Selection_mode
			Canvas.set_pointer_style 
				((create {EV_STOCK_PIXMAPS}).Standard_cursor)

			selection_button.select_actions.block
			move_button.select_actions.block
			information_button.select_actions.block
			
			if not selection_button.is_selected then
				selection_button.enable_select
			end

			move_button.disable_select
			information_button.disable_select

			selection_button.select_actions.resume
			move_button.select_actions.resume
			information_button.select_actions.resume
		ensure
			mode_set: mode = Selection_mode
		end

	set_move_mode is
			-- Set map move mode.
		do
			mode := Move_mode
			Canvas.set_pointer_style (move_cursor)

			selection_button.select_actions.block
			move_button.select_actions.block
			information_button.select_actions.block
			
			if not move_button.is_selected then
				move_button.enable_select
			end

			selection_button.disable_select
			information_button.disable_select

			selection_button.select_actions.resume
			move_button.select_actions.resume
			information_button.select_actions.resume
		ensure
			mode_set: mode = Move_mode
		end

	set_information_mode is
			-- Set information mode.
		do
			mode := Information_mode
			Canvas.set_pointer_style (information_cursor)

			selection_button.select_actions.block
			move_button.select_actions.block
			information_button.select_actions.block
			
			if not information_button.is_selected then
				information_button.enable_select
			end

			selection_button.disable_select
			move_button.disable_select

			selection_button.select_actions.resume
			move_button.select_actions.resume
			information_button.select_actions.resume
		ensure
			mode_set: mode = Information_mode
		end

feature {MAIN_CONTROLLER} -- Access

	mode: INTEGER
			-- Interface mode

feature {MAIN_CONTROLLER} -- Element change

	set_map_loaded (loaded: BOOLEAN) is
			-- `True' if map loaded
		do
			map_loaded := loaded
		ensure
			map_loaded_set: map_loaded = loaded
		end

feature -- Status report

	map_loaded: BOOLEAN

	is_in_default_state: BOOLEAN is
			-- Is the window in its default state?
			-- (as stated in `initialize')
		do
			Result := --(width = Window_width) and then
				--(height = Window_height) and then
				(title.is_equal (Window_title)) and then
				selection_button /= Void and then
				move_button /= Void and then
				information_button /= Void 
		end

feature -- Conversion

	client_to_map_coordinates (x, y: INTEGER): REAL_COORDINATE is
			-- Map position corresponding to client coordinates (`x', `y')
		local
			lx: DOUBLE
			ly: DOUBLE
			xperc: DOUBLE
			yperc: DOUBLE
			h: INTEGER
			org: REAL_COORDINATE
		do
			lx := x / 1
			ly := y / 1

			xperc := lx / (canvas.parent.client_width)
			h := (canvas.parent.client_height).max (1)
			yperc := (h - ly) / h

			org := canvas.visible_area.lower_left
			create Result.make (
				(org.x + xperc * canvas.visible_area.width).rounded, 
				(org.y + yperc * canvas.visible_area.height).rounded)
				
		ensure
			Result_exists: Result /= Void
		end

end

--|--------------------------------------------------------
--| This file is Copyright (C) 2004 by ETH Zurich.
--|
--| For questions, comments, additions or suggestions on
--| how to improve this package, please write to:
--|
--|     Michela Pedroni <michela.pedroni@inf.ethz.ch>
--|
--|--------------------------------------------------------
