indexing
	description: 
		"[
			Simple interface to a line in the city
		 ]"
		 
	author: "Michela Pedroni, ETH Zurich"
	date: "$Date: 2004/10/20 10:03:03 $"
	revision: "$Revision: 1.4 $"

class SIMPLE_LINE inherit
	
	SHARED_FACTORY
	
	DISPLAYER_ACCESSOR

create
	make
	
feature -- Initialization

	make (a_line: LINE) is
			-- Create a simple line that interfaces `a_line'.
		require
			a_line_exists: a_line /= Void
			a_line_undirected: not a_line.directed
		do
			internal_line := a_line
			internal_place := internal_line.sw_to_ne_line.first_place
		ensure
			internal_line /= Void
			internal_line_set: internal_line = a_line
		end

feature -- Status report

	after: BOOLEAN is
			-- Is there no valid position to right of cursor?
		do
			Result := internal_after
		end
		
	before: BOOLEAN is
			-- Is there no valid position to left of cursor?
		do
			Result := internal_line.sw_to_ne_line.before
		end
		
	off: BOOLEAN is
			-- Is there no item at cursor position?
		do
			Result := after or before
		end
		
	is_empty: BOOLEAN is
			-- Is container empty?
		do
			Result := internal_line.sw_to_ne_line.is_empty
		end		
		
feature -- Access

	count: INTEGER is
			-- Number of stations that belong to this line.
		do
			Result := internal_line.sw_to_ne_line.count + 1
		end

	i_th (an_index: INTEGER): PLACE is
			-- Place at index `an_index'
		require
			not_too_small: 1 <= an_index
			not_too_large: an_index <= count
		do
			if not internal_line.sw_to_ne_line.is_empty then
				Result := internal_line.sw_to_ne_line.i_th_place (an_index)
			elseif an_index = 1 then
				Result := internal_place
			end
		end	
		
	item: PLACE is
			-- Place at current index
		require
			not_off: not off
		do
			if not internal_line.sw_to_ne_line.after then
				Result := internal_line.sw_to_ne_line.place_for_iteration
			elseif not after then
				Result := ne_end		
			end
		end
		
		
	sw_end: PLACE is
			-- South west end of the line
		do
			if not internal_line.sw_to_ne_line.is_empty then
				Result := internal_line.sw_to_ne_line.first_place
			else
				Result := internal_place
			end
		ensure
			sw_end_is_first: Result = i_th (1)
		end
		
	
	ne_end: PLACE is
			-- North east end of the line
		do
			if not internal_line.sw_to_ne_line.is_empty then
				Result := internal_line.sw_to_ne_line.last_place
			else 
				Result := internal_place
			end
		ensure
			ne_end_is_last: Result = i_th (count)
		end

feature -- Cursor movement

	start is
			-- Move `a_cursor' to first position.
			-- (Performance: O(1).)
		do
			internal_line.sw_to_ne_line.start
			internal_after := False
		end
		
	forth is
			-- Move cursor to next position.
		do
			if not internal_line.sw_to_ne_line.after then
				internal_line.sw_to_ne_line.forth
				internal_after := False
			else
				internal_after := True
			end
		end	
feature -- Element change

	extend (a_place: PLACE) is
			-- Extend the line with an undirected link to `a_place'.
		require
			a_place_exists: a_place /= Void
		local
			l: LINK
			pp: ARRAYED_LIST [COORDINATE]
			ld: LINK_DISPLAYER
		do
			create pp.make (2)
			pp.extend (create {COORDINATE}.make (ne_end.position.x, ne_end.position.y))
			pp.extend (create {COORDINATE}.make (a_place.position.x, a_place.position.y))
			factory.build_link (ne_end.name, a_place.name, False, internal_line.city, internal_line)
			l := factory.link
			l.set_polypoints (pp)
			create ld.make (l)
		ensure
			new_station_added: i_th (count) = a_place
			added_at_NE: ne_end = a_place
			one_more: count = old count + 1
		end
		
	
feature -- Removal

	remove_all_links is
			-- Remove all links from the line.
			-- Places stay in the system.
		do
			from
			until
				internal_line.sw_to_ne_line.count = 0
			loop
				internal_line.prune_link (internal_line.sw_to_ne_line.first_link)
			end
		ensure
			only_one_left: count = 1
			both_ends_same: ne_end = sw_end
		end		
		
feature {NONE} -- Implementation

	internal_line: LINE
			-- Line that this class gives a simpler interface for
			
	internal_place: PLACE
			-- South west end of the line
			
	internal_after: BOOLEAN
			-- Is the cursor after (for iterating over places)?

invariant
	
	internal_line_undirected: not internal_line.directed
	
end -- class SIMPLE_LINE

--|--------------------------------------------------------
--| This file is Copyright (C) 2004 by ETH Zurich.
--|
--| For questions, comments, additions or suggestions on
--| how to improve this package, please write to:
--|
--|     Michela Pedroni <michela.pedroni@inf.ethz.ch>
--|
--|--------------------------------------------------------
