indexing
	description: 
		"[
			Displayers for place objects.
		 ]"
		 
	author: "Michela Pedroni, ETH Zurich"
	date: "$Date: 2004/10/20 10:03:04 $"
	revision: "$Revision: 1.7 $"

class PLACE_DISPLAYER inherit
	
	ELEMENT_DISPLAYER
		rename 
			element as place
		redefine 
			place 
		end
		
	KL_SHARED_FILE_SYSTEM
	
create 
	make

feature -- Access
	
	place: PLACE
			-- Place that this displayer displays

	label: DRAWABLE_TEXT
			-- Label that shows the place name

	pixmap_file_name: STRING
			-- Filename for a picture of the place

	bounding_box: REAL_RECTANGLE
			-- Bounding box of the graphical representation (without the label)
		
feature -- Element change

	set_pixmap_file_name (a_file_name: STRING) is
			-- Set `pixmap_file_name' to `a_file_name'.
		require
			a_file_name_exists: a_file_name /= Void
			a_file_name_not_empty: not a_file_name.is_empty
		local
			abs_path: STRING
			e: EV_ENVIRONMENT
			ext: STRING
		do			 
			abs_path := File_system.absolute_pathname (File_system.pathname_from_file_system (a_file_name, Windows_file_system))
			create e
			ext := File_system.extension (abs_path)
			ext.remove (1)
			ext.to_upper
			if e.supported_image_formats.has (ext) and File_system.file_exists (abs_path) then
				pixmap_file_name := abs_path
			end	
		end

feature -- Constants

	Mark_diameter: INTEGER is 10
			-- Diameter that gets added to the smallest circle around the graphical representation for marking
	
	Default_mark_color: EV_COLOR is
			-- Default color for the marking circle
		once
			create Result.make_with_8_bit_rgb (0, 0, 255)
		end

	Default_standard_color: EV_COLOR is
			-- Default standard color of place
		once
			create Result.make_with_rgb (0.8, 0.8, 0.8)
		end
		
	Default_highlight_color: EV_COLOR is
			-- Default color of highlighted place
		once
			create Result.make_with_rgb (0.0, 1.0, 0.0)
		end
		
feature -- Basic operations

	show_name is
			-- Show the label with the name of the place
		require
			label_exists: label /= Void
		do
			Drawable_objects.extend (label)
		end
		
	hide_name is
			-- Hide the label with the name of the place
		require
			label_exists: label /= Void
			label_in_drawable_objects: Drawable_objects.has (label)
		do
			Drawable_objects.prune_all (label)
		end

	place_name (a_bounding_box_list: LINKED_LIST [REAL_RECTANGLE]) is
			-- Place label at a good position (with collision detection).
			-- `a_bounding_box_list' holds the bounding boxes of elements that should not overlap 
			-- The bounding box of the label gets added to the `a_bounding_box_list'.
		require
			a_bounding_box_list_exists: a_bounding_box_list /= Void
		local
			pos: REAL_COORDINATE
			dpos: REAL_COORDINATE
			bb: REAL_RECTANGLE
			i: INTEGER
			placed: BOOLEAN
			overlap: BOOLEAN
		do
			dpos := text_position (1)
			from
				i := 1
			until
				placed or i > 4
			loop
				pos := text_position (i)
				label.set_position (pos)
				bb := label.bounding_box
				from
					overlap := False
					a_bounding_box_list.start
				until
					overlap or a_bounding_box_list.after
				loop
					overlap := bb.intersects (a_bounding_box_list.item)
					a_bounding_box_list.forth
				end
				
				placed := not overlap
				i := i + 1
			end

			if not placed then
				label.set_position (dpos)
			end
			a_bounding_box_list.extend (label.bounding_box)
		ensure
			a_bounding_box_list_one_more: a_bounding_box_list.count = old a_bounding_box_list.count + 1
		end

	adjust_bounding_box (a_coordinate: COORDINATE) is
			-- Adjust upper_right and lower_left to hold `a_coordinate'.
		local
			ll, ur: REAL_COORDINATE			
		do
			if lower_left = Void then
				create lower_left
				lower_left.set_x (a_coordinate.x)
				lower_left.set_y (a_coordinate.y)
			else
				if a_coordinate.x < lower_left.x then
					lower_left.set_x (a_coordinate.x)
				end
				if a_coordinate.y < lower_left.y then
					lower_left.set_y (a_coordinate.y)
				end
			end
			if upper_right = Void then
				create upper_right
				upper_right.set_x (a_coordinate.x)
				upper_right.set_y (a_coordinate.y)
			else
				if a_coordinate.x > upper_right.x then
					upper_right.set_x (a_coordinate.x)
				end
				if a_coordinate.y > upper_right.y then
					upper_right.set_y (a_coordinate.y)
				end
			end
			create ll.make (lower_left.x, lower_left.y)
			create ur.make (upper_right.x, upper_right.y)
			calculate_center
		ensure
			a_coordinate_smaller_than_upper_right: a_coordinate.x <= upper_right.x and a_coordinate.y <= upper_right.y
			a_coordinate_larger_than_lower_left: a_coordinate.x <= upper_right.x and a_coordinate.y <= upper_right.y
		end

feature -- Status report

	has_point (a_coordinate: REAL_COORDINATE): BOOLEAN is
			-- Is `a_coordinate' on the graphical representation of the place?
		do
			if (lower_left.x-3 < a_coordinate.x) and (upper_right.x+3 > a_coordinate.x) and 
			(lower_left.y-3 < a_coordinate.y) and (upper_right.y+3 > a_coordinate.y) then
				Result := True
			end				
		end

feature {NONE} -- Implementation

	initialize is
			-- Fill the `basic_representation', `marked_representation', and `highlighted_representation' with the right drawables.
		local
			pm: DRAWABLE_PIXMAP
			rr: DRAWABLE_ROUNDED_RECTANGLE
			c: DRAWABLE_CIRCLE
		do
			if pixmap_file_name /= Void then
				pm := pixmap (pixmap_file_name)
				basic_representation.extend (pm)
				highlighted_representation.extend (pixmap (pixmap_file_name))
				
				marked_representation.extend (circle (diameter + Mark_diameter, False, Default_mark_color))
				bounding_box := pm.bounding_box
			else
				if lower_left /= Void and upper_right /= Void then
					rr := rectangle (Standard_color, True)
					bounding_box := rr.bounding_box					
					basic_representation.extend (rr)
					highlighted_representation.extend (rectangle (Highlight_color, True))
					marked_representation.extend (rectangle (Default_mark_color, False))
				else
					c := circle (10, True, standard_color)
					basic_representation.extend (c)
					highlighted_representation.extend (circle (12, True, highlight_color))
					marked_representation.extend (circle (10 + Mark_diameter, False, Default_mark_color))
					bounding_box := c.bounding_box
				end
				-- recalculate bounding box
			end
			create label.make (place.name, create {REAL_COORDINATE}.make (place.position.x, place.position.y))
		ensure then
			bounding_box_set: bounding_box /= Void
		end

	lower_left, upper_right: COORDINATE
			-- The two edges that mark the bounding box of the graphical representation (without the label)

	circle (a_diameter: INTEGER; filled: BOOLEAN; a_color: EV_COLOR): DRAWABLE_CIRCLE	is
			-- Return a circle at position of the place with diameter `a_diameter' and color `a_color'
		require
			a_diameter_greater_than_zero: a_diameter > 0
			a_color_exists: a_color /= Void
		do
			create Result.make (create {REAL_COORDINATE}.make (place.position.x, place.position.y))
			if filled then
				Result.set_filled			
			end
			Result.set_color (a_color)
			Result.set_diameter (a_diameter)
		ensure
			Result_exists: Result /= Void
		end
		
	rectangle (a_color: EV_COLOR; filled: BOOLEAN): DRAWABLE_ROUNDED_RECTANGLE is
			-- Return a filled rectangle at position of the place with color `a_color'
		require
			a_color_exists: a_color /= Void			
		local
			ll, ur: REAL_COORDINATE
		do
			ll := real_coordinate (lower_left) + (create {REAL_COORDINATE}.make (-3, -3))
			ur := real_coordinate (upper_right) + (create {REAL_COORDINATE}.make (3, 3))
			create Result.make (ll, ur)
			Result.set_color (a_color)
			Result.set_edge_color (a_color)
			if not filled then
				Result.set_unfilled
			end
		ensure
			Result_exists: Result /= Void
		end
		
	diameter: INTEGER is
			-- Diameter that fits into the rectangle are between `lower_left' and `upper_right'
		require
			upper_right_exists: upper_right /= Void
			lower_left_exists: lower_left /= Void
		do
			if upper_right.y - lower_left.y < upper_right.x - lower_left.x then
				Result := upper_right.x - lower_left.x
			else
				Result := upper_right.y - lower_left.y
			end	
			Result := Result + 2
			Result := Result.max (10)
		end
		
	pixmap (a_file_name: STRING): DRAWABLE_PIXMAP is
			-- Return a drawable pixmap (from the file with name `a_file_name') that is positioned at the position of the place 
		require
			a_file_name_exists: a_file_name /= Void
			a_file_name_not_empty: not a_file_name.is_empty
		local
			x1, x2: DOUBLE
			y1, y2: DOUBLE
			pic: EV_PIXMAP
			c: REAL_RECTANGLE
		do
			x1 := place.position.x
			y1 := place.position.y
			x2 := place.position.x
			y2 := place.position.y
			create pic
			pic.set_with_named_file (a_file_name)
			x1 := x1 - (pic.width / 2)
			y1 := y1 + (pic.height / 2)
			x2 := x2 + (pic.width /2)
			y2 := y2 - (pic.height /2)
			create c.make_from_reals (x1.truncated_to_real, y1.truncated_to_real, x2.truncated_to_real, y2.truncated_to_real)
			create Result.make (pic, create {REAL_COORDINATE}.make (x1, y2))
			adjust_bounding_box (create {COORDINATE}.make (x1.truncated_to_integer, y1.truncated_to_integer))
			adjust_bounding_box (create {COORDINATE}.make (x2.truncated_to_integer, y2.truncated_to_integer))	
		end

	text_position (i: INTEGER): REAL_COORDINATE is
			-- `i'-th position for the label text
		local
			nx: DOUBLE
			ny: DOUBLE
		do
			inspect i
			when 1 then -- lower right corner of the place
				nx := bounding_box.upper_right.x + 1
				ny := bounding_box.upper_right.y + 3 + label.text_height
			when 2 then -- upper right corner of the place
				nx := bounding_box.upper_right.x + 1
				ny := bounding_box.lower_left.y - 1
			when 3 then -- lower left corner of the place
				nx := bounding_box.lower_left.x - label.text_width - 1
				ny := bounding_box.lower_left.y - 1
			when 4 then -- upper left corner of the place
				nx := bounding_box.lower_left.x - label.text_width - 1
				ny := bounding_box.upper_right.y + 3 + label.text_height
			else
				nx := bounding_box.upper_right.x + 1
				ny := bounding_box.lower_left.y - 1
			end

			create Result.make (nx.truncated_to_real, ny.truncated_to_real)
		ensure
			Result_exists: Result /= Void
		end

	calculate_center is
			-- Recalculate the center of the place and set the position of the place to it.
		local
			c: COORDINATE
		do
			create c.default_create
			c.set_x (((lower_left.x + upper_right.x) / 2).truncated_to_integer)
			c.set_y (((lower_left.y + upper_right.y) / 2).truncated_to_integer)
			place.set_position (c)
		end
		
end -- class PLACE_DISPLAYER

--|--------------------------------------------------------
--| This file is Copyright (C) 2004 by ETH Zurich.
--|
--| For questions, comments, additions or suggestions on
--| how to improve this package, please write to:
--|
--|     Michela Pedroni <michela.pedroni@inf.ethz.ch>
--|
--|--------------------------------------------------------
