indexing
	description: 
		"[
			Canvas with comfortable zooming and panning abilities.
		 ]"
		 
	author: "Rolf Bruderer & Michela Pedroni, ETH Zurich"
	date: "$Date: 2004/10/20 10:03:03 $"
	revision: "$Revision: 1.6 $"

class TRAFFIC_CANVAS inherit 

	CANVAS
		redefine 
			make, 
			set_size 
		end

create 
	make
	
feature -- Canvas elements

	make is
			-- Initialize.
		do
			Precursor
			set_zoom_limits (0.5, 2)
			zoom_factor := 1.0
			has_pending_redraw := false
			force_redraw_agent := agent force_redraw_agent_impl			
		end
		
	zoom_factor: REAL
			-- The zooming factor

	zoom_maximum: REAL
			-- Max zoom out factor

	zoom_minimum: REAL
			-- Max zoom in factor
			
	set_zoom_limits (min, max: REAL) is
			-- Set the both zoom limits `zoom_minimum' and `zoom_maximum'.
		require
			min_smaller_than_max: min < max
			min_greater_than_zero: min > 0
		do
			zoom_minimum := min
			zoom_maximum := max
		end		
		
	force_redraw is
			-- Redraw the map immediately.
		do			
			-- If there are any objects, then repaint.
			if object_list /= Void then
				refresh	
			end			
		end		
	
	has_pending_redraw : BOOLEAN 
			-- Is there any pending redraw?
	
	redraw is
			-- Redraw the map as soon as possible.
		do
			if not has_pending_redraw then
				(create {EV_ENVIRONMENT}).application.idle_actions.extend(force_redraw_agent)
				has_pending_redraw := true			
			end			
		ensure
			redraw_pending: has_pending_redraw
		end	
		
	update_visible_area is
			-- Updates the visible area according to `width', `height' and `zoom_factor'.
		local
			new_visible_area: REAL_RECTANGLE
		do
			-- Create new visible area with `width' and `height'.
			create new_visible_area.make (
				create {REAL_COORDINATE}.make(0.0, 0.0),
				create {REAL_COORDINATE}.make(width, height)
			)

			-- Zoom new visible area.
			new_visible_area := new_visible_area.scale (zoom_factor)
			
			-- Ensure that center of zoomed new visible area is same as old visible area.			
			new_visible_area := new_visible_area.up_by (visible_area.center.y - new_visible_area.center.y)
			new_visible_area := new_visible_area.right_by (visible_area.center.x - new_visible_area.center.x)
			
			-- Set new visible area.
			set_visible_area (new_visible_area)							
			
			-- Redraw the canvas as soon as possible.
			redraw
			
		end
		
	zoom_in (zoom_factor_delta: REAL) is
			-- Zoom in stepwise.
		do		
			zoom_factor := (zoom_factor - zoom_factor_delta).max (zoom_minimum)			
			update_visible_area	
		end
		
	zoom_out (zoom_factor_delta: REAL) is
			-- Zoom out stepwise. 
		do
			zoom_factor := (zoom_factor + zoom_factor_delta).min (zoom_maximum)
			update_visible_area			
		end

	go_right (pan_distance: REAL) is
			-- Move the map right.
		do
			visible_area := visible_area.left_by (pan_distance)
			redraw
		end
		
	go_left (pan_distance: REAL) is
			-- Move the map left.	
		do
			visible_area := visible_area.right_by (pan_distance)
			redraw
		end
	
	go_up (pan_distance: REAL) is
			-- Move the map up.	
		do
			visible_area := visible_area.down_by (pan_distance)
			redraw
		end
		
	go_down (pan_distance: REAL) is
			-- Move the map down.	
		do
			visible_area := visible_area.up_by (pan_distance)
			redraw
		end

	set_size (a_width, a_height: INTEGER) is
			-- Assign `a_width' and `a_height' to `width' and `weight'.
			-- Keep `zoom_factor' and keep center position.
		do
			Precursor (a_width, a_height)
			update_visible_area						
		end
	
feature {NONE} -- Implementation

	force_redraw_agent: PROCEDURE [ANY, TUPLE]
			-- Redraw agent to redraw the map when idle
	
	force_redraw_agent_impl is
			-- Redraw the map when idle.
		do
			force_redraw
			has_pending_redraw := false
			(create {EV_ENVIRONMENT}).application.idle_actions.prune_all (force_redraw_agent)			
		end
	
end -- class SPECIAL_CANVAS

--|--------------------------------------------------------
--| This file is Copyright (C) 2004 by ETH Zurich.
--|
--| For questions, comments, additions or suggestions on
--| how to improve this package, please write to:
--|
--|     Michela Pedroni <michela.pedroni@inf.ethz.ch>
--| 	Rolf Bruderer <bruderer@computerscience.ch>
--|
--|--------------------------------------------------------
