indexing
	description:
		"[
			Displayer for city objects.
		 ]"
		
	author: "Marcel Kessler & Michela Pedroni, ETH Zurich"
	date: "$Date: 2004/10/20 10:03:04 $"
	revision: "$Revision: 1.5 $"

class CITY_DISPLAYER inherit
	
	SHARED_DISPLAYER
	
	DISPLAYER
		rename 
			element as city
		redefine 
			city, 
			make 
		end
		
	KL_SHARED_FILE_SYSTEM
	
	DISPLAYER_ACCESSOR
	
create 
	make
			
feature -- Initialization
		
	make (a_city: CITY) is
			-- Creation procedure.
		do
			Precursor (a_city)
			show_names
			create background_polygon_list.make
			create bounding_box_list.make
		end
		
feature -- Access

	city: CITY
			-- City that the displayer displays

	bounding_box_list: LINKED_LIST [REAL_RECTANGLE]
			-- List of bounding boxes for calculation of name display
	
	background_polygon_list: LINKED_LIST [DRAWABLE_POLYGON]
			-- List of all the background polygons	

	background_file_name: STRING
			-- Absolute pathname for the background pic
			
feature -- Status report

	names_on: BOOLEAN
			-- Are the names of places displayed?

feature -- Element change

	set_background_file_name (a_file_name: STRING) is 
			-- Set `background_file_name' to `a_file_name' if it is a valid image filename.
		local
			abs_path: STRING
			e: EV_ENVIRONMENT
			ext: STRING
		do
			abs_path := File_system.absolute_pathname (a_file_name)
			create e
			ext := File_system.extension (abs_path)
			ext.remove (1)
			ext.to_upper
			if e.supported_image_formats.has (ext) then
				background_file_name := abs_path
			end	
		end
		
feature -- Basic operations

	build is
			-- Build the city.
		do 
			bounding_box_list.wipe_out
			drawable_objects.wipe_out
			calculate_places_bounding_box
			draw_background
			draw_lines
			draw_places
			draw_place_names
		end	
	
feature -- Status setting

	show_names is
			-- Show names on map.
		do
			from
				city.places.start
			until
				city.places.off
			loop
				if city.places.item_for_iteration.has_displayer then
					set_place_displayer (city.places.item_for_iteration)
					place_displayer.show_name					
				end
				city.places.forth
			end
			names_on := True
		end

	hide_names is
			-- Hide names on map.
		do
			from
				city.places.start
			until
				city.places.off
			loop
				if city.places.item_for_iteration.has_displayer then
					set_place_displayer (city.places.item_for_iteration)
					place_displayer.hide_name					
				end
				city.places.forth
			end
			names_on := False
		end

feature -- Status report

	place_at_coordinate (a_coordinate: REAL_COORDINATE): PLACE is
			-- Select place at `a_coordinate', if any.
		require
			point_exists: a_coordinate /= Void
		do
			from
				city.places.start
			until
				city.places.after or Result /= Void
			loop
				if city.places.item_for_iteration.has_displayer then
					set_place_displayer (city.places.item_for_iteration) 
					if place_displayer.has_point (a_coordinate) then
						Result := city.places.item_for_iteration
					end
				end
				city.places.forth
			end	
		end

	bounding_box: EV_RECTANGLE is
			-- Find the bounding box of the map in order to center it after loading
		local
			x, y: INTEGER
		do
			from
				city.places.start
			until
				city.places.after
			loop
				x := x.max (city.places.item_for_iteration.position.x)
				y := y.max (city.places.item_for_iteration.position.y)
				city.places.forth
			end
			
			create Result.make (0, -y, x, y)
		end
		
feature {NONE} -- Implementation

	display_names: BOOLEAN
			-- Are place names displayed?

	draw_background is
			-- Draw the background (using a .png file and/or polygons)
		local
			bg: DRAWABLE_PIXMAP
			pm: EV_PIXMAP
		do
			if background_file_name /= Void then
				create pm.default_create
				pm.set_with_named_file (background_file_name)
				create bg.make(pm, create {REAL_COORDINATE}.make (0, 0))
				insert_command (bg)				
			end
			insert_list (background_polygon_list)
		end
	
	draw_places is
			-- Draw the places as grey spots.
		do		
			from 
				city.places.start
			until
				city.places.off
			loop
				if city.places.item_for_iteration.has_displayer then
					set_place_displayer (city.places.item_for_iteration)
					place_displayer.reinitialize
					place_displayer.show
					bounding_box_list.extend (place_displayer.bounding_box)
				end
				city.places.forth
			end
		end
		
	draw_place_names is
			-- Write the names of the places next to them.
		do
			if names_on then
				from 
					city.places.start
				until
					city.places.off
				loop
					if city.places.item_for_iteration.has_displayer then
						set_place_displayer (city.places.item_for_iteration)
						place_displayer.place_name (bounding_box_list)
						place_displayer.show_name
					end
					city.places.forth
				end
			end
		end

	draw_lines is
			-- Draw the links of all the lines.
		do
			from
				city.lines.start
			until
				city.lines.off
			loop
				if city.lines.item_for_iteration.has_displayer then
					set_line_displayer (city.lines.item_for_iteration)
					line_displayer.reinitialize
					line_displayer.show					
				end
				city.lines.forth
			end
		end

	calculate_places_bounding_box is
			-- For each place calculate a rectangle that has center on position of the place
			-- and covers all links that lead to or from the place
		local
			pd: PLACE_DISPLAYER
		do	
			from
				city.links.start
			until
				city.links.off
			loop
				if city.links.item_for_iteration.from_place.has_displayer then
					set_place_displayer (city.links.item_for_iteration.from_place)
					pd := place_displayer
				end
				if pd /= Void then
					pd.adjust_bounding_box (city.links.item_for_iteration.polypoints.first)	
				end
				pd := Void
				if city.links.item_for_iteration.to_place.has_displayer then
					set_place_displayer (city.links.item_for_iteration.to_place)
					pd := place_displayer
				end
				if pd /= Void then
					pd.adjust_bounding_box (city.links.item_for_iteration.polypoints.last)	
				end
				city.links.forth
			end
		end
	
invariant

	bounding_box_list_exists: bounding_box_list /= Void
	background_polygon_list_exists: background_polygon_list /= Void
	
end -- class CITY_DISPLAYER

--|--------------------------------------------------------
--| This file is Copyright (C) 2004 by ETH Zurich.
--|
--| For questions, comments, additions or suggestions on
--| how to improve this package, please write to:
--|
--|     Michela Pedroni <michela.pedroni@inf.ethz.ch>
--|
--|--------------------------------------------------------
