indexing
	description: 
		"[
			Objects that traverse a graph in a 'something-first-order'.
		 ]"
		 
	author: "Michela Pedroni, ETH Zurich"
	date: "$Date: 2004/10/20 10:03:04 $"
	revision: "$Revision: 1.2 $"

deferred class GRAPH_TRAVERSER [G -> HASHABLE, H]

feature -- Access

	make (a_graph: GRAPH [G, H]) is
			-- Create a new traverser that is traversing `a_graph'. The starting position
			-- needs to be set with `start_with'.
		require
			non_void: a_graph /= Void
		do
			graph := a_graph
			create visited_nodes.make (a_graph.nodes_count)
			create_dispenser
		ensure
			graph_set: graph = a_graph
		end
		
	item: NODE [G, H]
			-- Node that the traverser currently points to

	graph: GRAPH [G, H]
			-- Graph that is to be traversed

feature -- Status report

	after: BOOLEAN 
			-- Is the cursor off?
	
feature -- Element change

	start_with (a_start_key: G) is
			-- Move the cursor back to the start at node with `a_start_key'.
		require
			a_start_key_exists: a_start_key /= Void
			graph_set: graph /= Void
			a_start_key_in_graph: graph.has_node (a_start_key)
		do
			item := graph.nodes.item (a_start_key)
			visited_nodes.wipe_out
			visited_nodes.put(a_start_key)
			dispenser.wipe_out
			add_reachable_targets (a_start_key)
			after := dispenser.is_empty
		ensure
			item_has_a_start_key: item.data = a_start_key
		end

	forth is
			-- Move the cursor to the next item.
		do
			if dispenser.is_empty then
				after := True
			else
				from
				until
					dispenser.is_empty or else (not visited_nodes.has(dispenser.item))					
				loop
					dispenser.remove
				end
				after := dispenser.is_empty or else (visited_nodes.has(dispenser.item))
				if not after then
					visited_nodes.put(dispenser.item)
					add_reachable_targets (dispenser.item)
					item := graph.nodes.item (dispenser.item)
				end
			end
		end

feature {BF_GRAPH_TRAVERSER, DF_GRAPH_TRAVERSER} -- Basic operations

	create_dispenser is
			-- Create the dispenser.
		deferred
		ensure
			dispenser_created: dispenser /= Void
		end

feature {NONE} -- Implementation

	visited_nodes: DS_HASH_SET [G]
			-- Nodes that are already visited by the traverser

	dispenser: DS_DISPENSER[G]
			-- Storage of items that need to be processed
	
	add_reachable_targets (a_key: G) is
			-- Add all nodes that are reachable from `a_key' to the `dispenser'.
		do
			from
				graph.nodes.item (a_key).edges.start
			until
				graph.nodes.item (a_key).edges.off
			loop
				dispenser.put (graph.nodes.item (a_key).edges.item_for_iteration.target_key)
				graph.nodes.item (a_key).edges.forth
			end			
		end		
	
invariant
	
	graph_set: graph /= Void
	
end -- class GRAPH_TRAVERSER

--|--------------------------------------------------------
--| This file is Copyright (C) 2004 by ETH Zurich.
--|
--| For questions, comments, additions or suggestions on
--| how to improve this package, please write to:
--|
--|     Michela Pedroni <michela.pedroni@inf.ethz.ch>
--|
--|--------------------------------------------------------
