indexing
	description: 
		"[
			Route calculator that given a network and a source and target place returns the shortest path as a ROUTE object.
		 ]"
		 
	author: "Michela Pedroni, ETH Zurich"
	date: "$Date: 2004/10/18 05:09:34 $"
	revision: "$Revision: 1.2 $"

class
	ROUTE_CALCULATOR

create 
	make
	
feature -- Initialization

	make (a_network: NETWORK) is
			-- Set `network' to `a_network'. The standard weight and criteria function is used for the route calculation.
		require
			a_network_exists: a_network /= Void
		do
			weight_function := agent link_length (?)
			validity_criterium := agent any_link (?)
			network := a_network
		end
		
feature -- Access

	weight_function: FUNCTION [ANY, TUPLE [LINK], DOUBLE]
			-- Function to decide on the wheight of a given link
	
	validity_criterium: FUNCTION [ANY, TUPLE [LINK], BOOLEAN]		
			-- Function to decide whether a given link is allowed to the shortest path calculation

	shortest_path_calculator: SHORTEST_PATH_CALCULATOR [PLACE, LINK]
			-- Calculator that takes over the calculation of a shortest path

feature -- Weight functions setters

	set_weight (a_weight_function: FUNCTION [ANY, TUPLE [LINK, LINK], DOUBLE]) is
			-- Set weight function to `a_weight_function'.
		do
			weight_function := a_weight_function
		end
		
	set_hop_count_weight is
			-- Set weight function to be `hop_count'.
		do
			weight_function := agent hop_count (?)
		end

	set_link_length_weight is
			-- Set weight function to be the `link_length'.
		do
			weight_function := agent link_length (?)
		end

feature -- Shortest path calculators setters

	set_shortest_path_calculator (a_path_calculator: SHORTEST_PATH_CALCULATOR [PLACE, LINK]) is
			-- Set `shortest_path_calculator' to `a_path_calculator'.
		require
			a_path_calculator_exists: a_path_calculator /= Void
		do
			shortest_path_calculator := a_path_calculator 
		end
	
	set_dijkstra_shortest_path_calculator is
			-- Use the Dijkstra algorithm for shortest path calculations
		require
			weight_function_is_set: weight_function /= Void
			validity_criterium_is_set: validity_criterium /= Void
		do
			shortest_path_calculator := create {DIJKSTRA_SHORTEST_PATH_CALCULATOR [PLACE, LINK]}.make (network)
		end
		
feature -- Basic operations

	calculate_route (a_places_to_visit_list: LINKED_LIST [PLACE]): ROUTE is
			-- Route that visits all places in `a_places_to_visit_list' (in the given order)
		require 
			calculator_exists: shortest_path_calculator /= Void
			enough_places_to_visit: a_places_to_visit_list.count >= 2
		local
			p1, p2: PLACE
			s: SHORTEST_PATH [PLACE, LINK]
		do
			create Result.make (a_places_to_visit_list)
			from
				a_places_to_visit_list.start
				p1 := a_places_to_visit_list.item
				a_places_to_visit_list.forth
			until
				a_places_to_visit_list.off
			loop
				p2 := a_places_to_visit_list.item
				shortest_path_calculator.reinitialize (weight_function, validity_criterium)
				shortest_path_calculator.calculate_shortest_path (p1, p2)
				s := shortest_path_calculator.shortest_path
				from
					s.start
				until
					s.off
				loop
					Result.links.extend (s.item.data)
					Result.places_on_route.extend (s.item.source_key)
					if s.islast then
						Result.places_on_route.extend (s.item.target_key)
					end
					s.forth
				end
				p1 := p2
				a_places_to_visit_list.forth
			end
		end

feature -- Weight functions

	link_length (a_link: LINK): DOUBLE is
			-- Link length 
		require
			a_link_exists: a_link /= Void
		do
			Result := a_link.length
		end
	
	hop_count (a_link: LINK): DOUBLE is
			-- Hop count
		require
			a_link_exists: a_link /= Void
		do
			Result := 1.0
		end
		
feature -- Criteria functions

	any_link (a_link: LINK): BOOLEAN is
			-- Any link is allowed to the route calculation
		require
			a_link_exists: a_link /= Void
		do
			Result := True
		end

feature {NONE} -- Implementation

	network: NETWORK
			-- Network on which the route calculation is done
			
invariant
	
	network_set: network /= Void

end -- class ROUTE_CALCULATOR

--|--------------------------------------------------------
--| This file is Copyright (C) 2004 by ETH Zurich.
--|
--| For questions, comments, additions or suggestions on
--| how to improve this package, please write to:
--|
--|     Michela Pedroni <michela.pedroni@inf.ethz.ch>
--|
--|--------------------------------------------------------
