indexing
	description: 
		"[
			Links in a city that link two places together. 
			Links may be directed or undirected and of a certain type (see LINE_LINK_TYPE_CONSTANTS).
		 ]"
		 
	author: "Michela Pedroni, ETH Zurich"
	date: "$Date: 2004/10/20 10:03:03 $"
	revision: "$Revision: 1.4 $"

class LINK inherit 

	LINE_LINK_TYPE_CONSTANTS
	
	CITY_ELEMENT

create
	make, make_walk, make_bus, make_metro
	
feature {CITY} -- Initialization

	make (a_type: STRING; a_directed: BOOLEAN; a_place1: PLACE; a_place2: PLACE) is
			-- Initialize the link to be of type `a_type', directed if `a_directed' is True, starting from place `a_place1' and going to place `a_place2'.
		require
			valid_type: is_valid_type (a_type)
			a_place1_exists: a_place1 /= Void
			a_place2_exists: a_place2 /= Void
			a_place1_not_same_as_a_place2: a_place1 /= a_place2
		do
			set_type (a_type)
			directed := a_directed
			from_place := a_place1
			to_place := a_place2
		end

	make_walk (a_place1: PLACE; a_place2: PLACE) is
			-- Initialize the link to be of type "walking", undirected, starting from place `a_place1' and going to place `a_place2'.
		require
			a_place1_exists: a_place1 /= Void
			a_place2_exists: a_place2 /= Void
			a_place1_not_same_as_a_place2: a_place1 /= a_place2			
		do
			make (walking_type, False, a_place1, a_place2)
		end
		
	make_metro (a_place1: PLACE; a_place2: PLACE) is
			-- Initialize the link to be of type "metro", undirected, starting from place `a_place1' and going to place `a_place2'.
		require
			a_place1_exists: a_place1 /= Void
			a_place2_exists: a_place2 /= Void
			a_place1_not_same_as_a_place2: a_place1 /= a_place2			
		do
			make (metro_type, False, a_place1, a_place2)
		end
			
	make_bus (a_place1: PLACE; a_place2: PLACE) is
			-- Initialize the link to be of type "bus", undirected, starting from place `a_place1' and going to place `a_place2'.
		require
			a_place1_exists: a_place1 /= Void
			a_place2_exists: a_place2 /= Void
			a_place1_not_same_as_a_place2: a_place1 /= a_place2			
		do
			make (bus_type, False, a_place1, a_place2)
		end
				
feature -- Access

	line: LINE
			-- Line to which this link belongs to
	
	type: STRING
			-- Type of this link 
	
	from_place: PLACE
			-- Start place of link 
	
	to_place: PLACE
			-- End place of link
		
	length: DOUBLE is
			-- Length of link
		local
			i: INTEGER
		do
			check
				polypoints.count >= 2
			end
			from
				i := 1
			until
				i >= polypoints.count
			loop
				Result := Result + polypoints.i_th (i).distance (polypoints.i_th (i+1))
				i := i + 1
			end
		end

	other_end (an_end: PLACE): PLACE is
			-- Not `an_end' end of link.
		require
			has (an_end)
		do
			if an_end = from_place then
				Result := to_place
			else
				Result := from_place
			end
		end

	polypoints: ARRAYED_LIST [COORDINATE]
			-- Polypoints of this link
		
feature -- Status setting
		
	set_type (a_type: STRING) is
			-- Set type of link to `a_type'.
		require
			a_type_exists: a_type /= Void
			a_type_is_valid: is_valid_type (a_type)
		do
			create type.make_from_string (a_type)	
		ensure
			type_set: type.is_equal (a_type)
		end

	set_line (a_line: LINE) is
			-- Set `line' to `a_line'.
		require
			a_line_exists: a_line /= Void
			a_line_fits: equal (a_line.type, type)
		do
			line := a_line
		ensure
			line_set: line = a_line
		end

	set_polypoints (a_list: ARRAYED_LIST [COORDINATE]) is
			-- Set `polypoints' to `a_list'.
		require
			a_list_exists: a_list /= Void
		do
			polypoints := a_list
		ensure
			polypoints_set: polypoints = a_list
		end
		
feature -- Status report	
			
	directed: BOOLEAN
			-- Is link directed?

	has (a_place: PLACE): BOOLEAN is
			-- Is `a_place' part of this link?
		do
			Result := (a_place = from_place) or (a_place = to_place)
		end

invariant
	
	line_has_same_type: line /= Void implies equal (line.type, type)
	
end -- class LINK

--|--------------------------------------------------------
--| This file is Copyright (C) 2004 by ETH Zurich.
--|
--| For questions, comments, additions or suggestions on
--| how to improve this package, please write to:
--|
--|     Michela Pedroni <michela.pedroni@inf.ethz.ch>
--|
--|--------------------------------------------------------
