indexing
	description: 
		"[ 
			Objects that represent a line on which transportation vehicles are moving along, e.g. metro lines, bus lines, but also streets.
			Note: For each direction of a line there is a one-way line that gives you the according links! 
		]"
		
	author: "Michela Pedroni, ETH Zurich"
	date: "$Date: 2004/10/20 10:03:03 $"
	revision: "$Revision: 1.5 $"

class LINE inherit
	
	LINE_LINK_TYPE_CONSTANTS
	
	CITY_ELEMENT
	
create
	make
	
feature -- Initialization

	make (a_name, a_type: STRING) is
			-- Set the name to `a_name' and the type to `a_type'.
		require
			a_name_exists: a_name /= Void
			a_type_exists: a_type /= Void
			a_type_valid: is_valid_type (a_type)
		do	
			create sw_to_ne_line.make
			create ne_to_sw_line.make
			set_name (a_name)
			set_type (a_type)
			directed := False
		ensure
			name_set: equal (name, a_name)
			type_set: equal (type, a_type)
		end
	
feature -- Access

	name: STRING
			-- Name of the line

	type: STRING	
			-- Type of the line (e.g. bus, metro, walking, etc.)

	sw_to_ne_line: ONE_WAY_LINE
			-- One-way line into the south west direction
	
	ne_to_sw_line: ONE_WAY_LINE
			-- One-way line into the north east direction
	
feature -- Status setting

	set_name (a_name: STRING) is
			-- Set name to `a_name'.
		require
			a_name_exists: a_name /= Void
		do
			create name.make_from_string (a_name)
		end
		
	set_type (a_name: STRING) is
			-- Set type to `a_name'.
		require
			a_name_exists: a_name /= Void 
			a_name_valid: is_valid_type (a_name)
		do
			create type.make_from_string (a_name)
		end

feature -- Status report

	has_link (a_link: LINK): BOOLEAN is
			-- Is `a_link' part of the line?
		require
			a_link_exists: a_link /= Void
		do
			Result := sw_to_ne_line.has_link (a_link) or ne_to_sw_line.has_link (a_link)
		end

	directed: BOOLEAN
			-- Is line directed?
		
feature -- Basic operations

	add_link (a_link: LINK) is
			-- Add link `a_link' to the line.
		require
			a_link_exists: a_link /= Void
			a_link_valid: is_valid_link_for_insertion (a_link.from_place, a_link.to_place, a_link.directed)
		do
			if not a_link.directed then
				if sw_to_ne_line.count = 0 then
					sw_to_ne_line.put_first (a_link, False)
				else
					sw_to_ne_line.put_end (a_link)
				end
				if ne_to_sw_line.count = 0 then
					ne_to_sw_line.put_first (a_link, True)
				else
					ne_to_sw_line.put_front (a_link)
				end
			elseif sw_to_ne_line.count < 1 or else (sw_to_ne_line.count > 0 and then sw_to_ne_line.last_place = a_link.from_place) then
				if sw_to_ne_line.count = 0 then
					sw_to_ne_line.put_first (a_link, False)
				else
					sw_to_ne_line.put_end (a_link)
				end
				directed := True
			elseif ne_to_sw_line.count < 1 or else (ne_to_sw_line.count > 0 and then ne_to_sw_line.first_place = a_link.to_place) then
				if ne_to_sw_line.count = 0 then
					ne_to_sw_line.put_first (a_link, True)
				else
					ne_to_sw_line.put_front (a_link)
				end
				directed := True
			end
		ensure
			link_added: has_link (a_link)
		end

	is_valid_link_for_insertion (a_place1, a_place2: PLACE; directed_link: BOOLEAN): BOOLEAN is
			-- Is a link going from `a_place1' to `a_place2' (directed or undirected) valid for insertion at end of `sw_to_ne_line' or at front of `ne_to_sw_line'?
		require
			a_place1_exists: a_place1 /= Void
			a_place2_exists: a_place2 /= Void
		do
			if not directed_link and ((sw_to_ne_line.count = 0) or (sw_to_ne_line.last_place = a_place1) or (sw_to_ne_line.last_place = a_place2)) and
			((ne_to_sw_line.count = 0) or (ne_to_sw_line.first_place = a_place1) or (ne_to_sw_line.first_place = a_place2)) then
				Result := True
			elseif (sw_to_ne_line.count = 0 or else sw_to_ne_line.last_place = a_place1) or (ne_to_sw_line.count = 0 or else ne_to_sw_line.first_place = a_place2) then
				Result := True
			end
		end
	
	is_valid_link_for_removal (a_link: LINK): BOOLEAN is
			-- 
		require
			a_link_exists: a_link /= Void
		do
			if not a_link.directed and ((a_link = sw_to_ne_line.last_link and a_link = ne_to_sw_line.first_link) or (a_link = sw_to_ne_line.first_link and a_link = ne_to_sw_line.last_link)) then
				Result := True
			elseif a_link = sw_to_ne_line.last_link or a_link = sw_to_ne_line.first_link or a_link = ne_to_sw_line.first_link or a_link = ne_to_sw_line.last_link then
				Result := True	
			end
		end
		
	prune_link (a_link: LINK) is
			-- Remove link `a_link' from the line and from the transportation network.
		require
			a_link_exists: a_link /= Void
			a_link_in_line: has_link (a_link)
			a_link_at_beginning_or_end: is_valid_link_for_removal (a_link)
		do
			if not a_link.directed then
				if a_link = sw_to_ne_line.first_link and a_link = ne_to_sw_line.last_link then
					sw_to_ne_line.prune_first_link
					ne_to_sw_line.prune_last_link
				elseif a_link = ne_to_sw_line.first_link and a_link = sw_to_ne_line.last_link then
					sw_to_ne_line.prune_last_link
					ne_to_sw_line.prune_first_link
				end
			else
				if a_link = sw_to_ne_line.first_link then
					sw_to_ne_line.prune_first_link
				elseif a_link = sw_to_ne_line.last_link then
					sw_to_ne_line.prune_last_link
				elseif a_link = ne_to_sw_line.first_link then
					ne_to_sw_line.prune_first_link
				elseif a_link = ne_to_sw_line.last_link then
					ne_to_sw_line.prune_last_link
				end
			end
			city.transport_network.prune_link (a_link)
		ensure
			not_in_line: not has_link (a_link)
		end
		
invariant
	
	sw_to_ne_line_exists: sw_to_ne_line /= Void
	ne_to_sw_line_exists: ne_to_sw_line /= Void
	
end -- class LINE

--|--------------------------------------------------------
--| This file is Copyright (C) 2004 by ETH Zurich.
--|
--| For questions, comments, additions or suggestions on
--| how to improve this package, please write to:
--|
--|     Michela Pedroni <michela.pedroni@inf.ethz.ch>
--|
--|--------------------------------------------------------
