indexing
	description: 
		"[
			Controller that steers the application's behavior and mediates between visual input/output and changes of the city model.
		 ]"
		
	author: "Marcel Kessler & Michela Pedroni, ETH Zurich"
	date: "$Date: 2004/10/20 10:03:04 $"
	revision: "$Revision: 1.8 $"

class MAIN_CONTROLLER inherit

	MAIN_WINDOW_CONSTANTS
		undefine default_create, copy end
	
	DISPLAYER_ACCESSOR
		undefine default_create, copy end
	
	SHARED_CITY
		undefine default_create, copy end

	EV_APPLICATION
	
create
	make_and_launch 

feature {NONE} -- Initialization

	make_and_launch is
			-- Initialize and launch application
		do
			default_create
			prepare
			launch
		end

	prepare is
			-- Prepare the first window to be displayed.
			-- Perform one call to first window in order to
			-- avoid to violate the invariant of class EV_APPLICATION.
		local
			acc: EV_ACCELERATOR
			k: EV_KEY_CONSTANTS
		do
			create application_directory.make_empty
			application_directory := File_system.absolute_pathname (File_system.current_working_directory)
			mouse_button := 0
			create main_window
			main_window.show
			
			main_window.close_request_actions.extend (agent request_close_window)
			main_window.file_close_item.select_actions.extend (agent request_close_window)
			main_window.file_open_item.select_actions.extend (agent request_open_city)
			create k
			create acc.make_with_key_combination
				(create {EV_KEY}.make_with_code (k.Key_o), True, False, False)
			acc.actions.extend (agent request_open_city)
			main_window.accelerators.extend (acc)
			main_window.name_display_button.select_actions.extend 
						(agent toggle_name_display)						
			main_window.canvas.mouse_wheel_actions.force_extend 
						(agent process_mouse_wheel (?))
			process_click_agent := (agent process_click (?, ?, ?, ?, ?, ?, ?, ?))
			main_window.canvas.pointer_button_press_actions.force_extend (process_click_agent)

			main_window.canvas.pointer_button_release_actions.force_extend 
						(agent process_release (?, ?, ?, ?, ?, ?, ?, ?))
			main_window.pointer_leave_actions.force_extend 
						(agent process_leave_window)
			main_window.viewport.resize_actions.force_extend
						(agent resize_canvas)
			main_window.help_about_item.select_actions.extend 
						(agent request_about)
			main_window.route_calculation_button.select_actions.extend 
						(agent request_calculate_route)
			create timeline.make
			timeline.set_canvas (main_window.canvas)
			create button_registry.make
			button_registry.set_up_examples
			button_registry.set_up_exercises
			main_window.button_panel.setup_examples_from_container (button_registry.examples)
			main_window.button_panel.setup_exercises_from_container (button_registry.exercises)
		end	

feature -- Access

	timeline: TIMELINE
			-- Timeline for route animations and other animations

	main_window: MAIN_WINDOW
			-- Main window for display
	
	mouse_button: INTEGER
			-- Number for distinguishing which mouse button was pressed (1: Left button, 2: Middle button, 3: Right button)
	
	selected_places: LINKED_LIST [PLACE]
			-- Places that are marked in the city

	route: ROUTE 
			-- Route that has been calculated
			
	button_registry: BUTTON_REGISTRY
	
	application_directory: STRING
			-- Absolute path where the application is located
	
feature -- Basic operations

	request_close_window is
			-- Cleanup before closing the application.
		do
			main_window.destroy
			destroy
		end

	request_about is
			-- Display the About dialog.
		local
			dlg: ABOUT_DIALOG
		do
			create dlg
			dlg.show_modal_to_window (main_window)
		end

	request_calculate_route is
			-- Calculate and display the route.
		local
			dlg: EV_INFORMATION_DIALOG
			sp: EV_STOCK_PIXMAPS
			obj: ANIMATABLE_SPOT
			route_calculator: ROUTE_CALCULATOR
			animation_generator: ROUTE_ANIMATION_GENERATOR
			n: NETWORK
		do
			create animation_generator
			if route /= Void then
				route.links.do_all (agent unhighlight_link (?))	
				route.places_on_route.do_all (agent unhighlight_place (?))
			end
			route := Void
			if main_window.map_loaded and selected_places.count >= 2 then
				create sp
				main_window.set_pointer_style (sp.Busy_cursor)
				n := city.transport_network
				create route_calculator.make (n)
				route_calculator.set_dijkstra_shortest_path_calculator
				route := route_calculator.calculate_route (selected_places)
				if route /= Void and not route.links.is_empty then
					route.links.do_all (agent highlight_link (?))
					route. places_on_route.do_all (agent highlight_place (?))
					timeline.reset
					animation_generator.build_route_animation (route)
					obj := animation_generator.animation_object
					timeline.extend (obj)
					timeline.add_stop_action (agent obj.unregister)
					set_city_displayer (city)
					timeline.add_stop_action (agent city_displayer.remove_command (obj))
					timeline.start
					main_window.canvas.refresh
					main_window.set_pointer_style (sp.Standard_cursor)
				else
					main_window.set_pointer_style (sp.Standard_cursor)
					create dlg.make_with_text ("No route found")
					dlg.show_modal_to_window (main_window)
				end
			end
		end

	request_open_city is
			-- Open the xml file of a city.
		local
			dlg: EV_FILE_OPEN_DIALOG
			fn: STRING
		do
			create dlg.make_with_title ("Open xml file...")
			dlg.show_modal_to_window (main_window)
			fn := dlg.file_name
			if fn /= Void and then not fn.is_empty and then File_system.file_exists (fn) then
				open_city (fn)
			end
		end

	open_city (a_filename: STRING) is
			-- 
		require
			a_filename_exists: a_filename /= Void
			a_filename_not_empty: not a_filename.is_empty
			a_file_exists: File_system.file_exists (a_filename)
		local
			cp: CITY_PARSER
			idlg: EV_INFORMATION_DIALOG
			sp: EV_STOCK_PIXMAPS
			adaptor: XML_ADAPTOR
		do
			create sp
			main_window.set_pointer_style (sp.Busy_cursor)
			create cp.make_with_factory (Factory)
			create adaptor
			adaptor.adapt_xml_registry (cp)
			cp.set_file_name (a_filename)
			cp.set_working_directory
			cp.parse
			main_window.traffic_console.append_text ("starting...%N")
			if cp.can_process then
				main_window.traffic_console.append_text ("parsing...%N")
				if not cp.has_error then
					cp.process
				end

				if not cp.has_error then
					set_city_displayer (city)
					city_displayer.build
					main_window.canvas.draw_all_items (city_displayer.drawable_objects)
					resize_canvas
					move_to_center
					main_window.set_map_loaded(True)
					create selected_places.make
					main_window.set_tool_bar_sensitivity (True)
					main_window.set_route_calculation_sensitivity (False)
					if not main_window.name_display_button.is_selected then
						main_window.name_display_button.toggle								
					end
				end
			elseif cp.has_error then  
				main_window.set_pointer_style (sp.Standard_cursor)
				create idlg.make_with_text (cp.error_description)
				idlg.show_modal_to_window (main_window)
				main_window.set_map_loaded(False)
				main_window.canvas.clear
				main_window.set_tool_bar_sensitivity (False)
			else
				main_window.set_pointer_style (sp.Standard_cursor)
				create idlg.make_with_text ("Could not process the xml-file. Check whether the root xml tag is correct.")
				idlg.show_modal_to_window (main_window)
				main_window.set_map_loaded (False)
				main_window.canvas.clear
				main_window.set_tool_bar_sensitivity (False)
			end			
				main_window.traffic_console.append_text ("done...%N")
			main_window.set_pointer_style (sp.Standard_cursor)
		end
		
feature -- Constants

	delta_t: REAL is 0.002
			-- The delta time between two timeline increments

feature {NONE} -- Implementation (Mouse events)

	last_cursor_position: REAL_COORDINATE
			-- Last remembered cursor position

	process_click_agent: PROCEDURE [ANY, TUPLE [INTEGER, INTEGER, INTEGER, DOUBLE, DOUBLE, DOUBLE, INTEGER, INTEGER]]
	
	process_click (x, y, b: INTEGER; x_t, y_t, p: DOUBLE; 
			scr_x, scr_y: INTEGER) is
			-- Clicking on the canvas.
			-- (For an explanation of arguments look at
			-- EV_POINTER_BUTTON_ACTION_SEQUENCE).
		local
			pt: REAL_COORDINATE
			pl: PLACE
		do
			if timeline.running then
				timeline.stop
			end
			mouse_button := b
			if main_window.map_loaded and then b = 1 then
				pt := main_window.client_to_map_coordinates (x, y)
				inspect
					main_window.mode
				when Selection_mode then
					set_city_displayer (city)
					pl := city_displayer.place_at_coordinate (pt)
					if pl /= Void then
						set_place_displayer (pl)
						if not place_displayer.marked then
							place_displayer.mark
							selected_places.extend (pl)
						else
							place_displayer.unmark
							selected_places.prune_all (pl)
						end
					else
						from
							selected_places.start
						until
							selected_places.off
						loop
							set_place_displayer (selected_places.item)
							if place_displayer.marked then
								place_displayer.unmark
							end
							selected_places.forth
						end
						selected_places.wipe_out
					end
					if route /= Void then
						route.links.do_all (agent unhighlight_link (?))
						route.places_on_route.do_all (agent unhighlight_place (?))
						route := Void
					end
					if selected_places.count >= 2 then
						main_window.set_route_calculation_sensitivity (True)
					else
						main_window.set_route_calculation_sensitivity (False)
					end
				when Information_mode then
					set_city_displayer (city)
					pl := city_displayer.place_at_coordinate (pt)
					if pl /= Void  and then pl.has_description then
						main_window.traffic_console.enable_edit
						main_window.traffic_console.append_text (pl.description + "%N")
						main_window.traffic_console.disable_edit
					end
				else
					-- Do nothing.
				end
			end
			if main_window.map_loaded and then (b = 3 or (main_window.mode = Move_mode and b = 1)) then
				pt := main_window.client_to_map_coordinates (x, y)
				last_cursor_position := pt
			end
			main_window.canvas.redraw
		end

	process_release (x, y, b: INTEGER; x_t, y_t, p: DOUBLE; 
			scr_x, scr_y: INTEGER) is
			-- Release mouse pointer.
			-- (For an explanation of arguments look at
			-- EV_POINTER_BUTTON_ACTION_SEQUENCE.)
		local
			pt: REAL_COORDINATE
			xdiff: DOUBLE
			ydiff: DOUBLE
		do
			if main_window.map_loaded then -- and main_window.mode = Move_mode then
				pt := main_window.client_to_map_coordinates (x, y)
				if last_cursor_position /= Void and ((main_window.mode = Move_mode and mouse_button = 1) or mouse_button = 3)  then					
					xdiff := last_cursor_position.x - pt.x
					ydiff := last_cursor_position.y - pt.y
					if xdiff /= 0 or ydiff /= 0 then
						main_window.canvas.go_down (ydiff)
						main_window.canvas.go_left (xdiff)
						main_window.canvas.redraw
					end
				end
				last_cursor_position := Void --main_window.client_to_map_coordinates (x, y)
			end
		end

	process_leave_window is
			-- Leave application window.
		do
			process_release (0, 0, 0, 0.0, 0.0, 0.0, 0, 0)
		end

	process_mouse_wheel (y: INTEGER) is
			-- Zoom in or out.
		do
			if main_window.map_loaded then
				if y > 0 then
					main_window.Canvas.zoom_in (y * Zoom_factor_stepwise)
				elseif y < 0 then
					main_window.Canvas.zoom_out (y * (-Zoom_factor_stepwise))
				end
			end
		end

feature {NONE} -- Implementation

		
	unhighlight_link (a_link: LINK) is
			-- Unhighlight `a_link' on the map.
		require
			a_link_exists: a_link /= Void
			a_link_has_displayer: a_link.has_displayer
		do
			set_link_displayer (a_link)
			link_displayer.unhighlight
		ensure
			link_not_highlighted: not link_displayer.highlighted 
		end

	highlight_link (a_link: LINK) is
			-- Highlight `a_link' on the map.
		require
			a_link_exists: a_link /= Void
			a_link_has_displayer: a_link.has_displayer
		do
			set_link_displayer (a_link)
			link_displayer.highlight
		ensure
			link_highlighted: link_displayer.highlighted 
		end
	
	unhighlight_place (a_place: PLACE) is
			-- Unhighlight `a_place' on the map.
		require
			a_place_exists: a_place /= Void
			a_place_has_displayer: a_place.has_displayer
		do
			set_place_displayer (a_place)
			place_displayer.unhighlight
		ensure
			place_not_highlighted: not place_displayer.highlighted 
		end

	highlight_place (a_place: PLACE) is
			-- Highlight `a_place' on the map.
		require
			a_place_exists: a_place /= Void
			a_place_has_displayer: a_place.has_displayer
		do
			set_place_displayer (a_place)
			place_displayer.highlight
		ensure
			place_highlighted: place_displayer.highlighted 
		end

	move_to_center is
			-- Center map on screen.
		local
			r: EV_RECTANGLE
			xdiff, ydiff: DOUBLE
			map_center, canvas_center: REAL_COORDINATE
		do
			set_city_displayer (city)
			r := city_displayer.bounding_box
			create map_center.make ((r.right - r.left)/2, (r.top - r.bottom)/2)
			canvas_center := main_window.client_to_map_coordinates ((main_window.canvas.width/2).floor, (main_window.canvas.height/2).floor)
			xdiff := map_center.x - canvas_center.x
			ydiff := (-1)*map_center.y - canvas_center.y
			if xdiff /= 0 or ydiff /= 0 then
				main_window.canvas.go_down (ydiff)
				main_window.canvas.go_left (xdiff) 
			end
			main_window.canvas.redraw
		end
		

	toggle_name_display is
			-- Toggle name display on/off.
		do
			set_city_displayer (city)
			if main_window.map_loaded then
				if main_window.name_display_button.is_selected then
					city_displayer.show_names
				else
					city_displayer.hide_names
				end
				main_window.canvas.refresh
			end
		end

	resize_canvas is
			-- Set up canvas.
		local
			w: INTEGER
			h: INTEGER
		do
			w := (main_window.viewport.width).max (1)  
			h := (main_window.viewport.height).max (1)
			main_window.canvas.set_size (w, h)
			main_window.canvas.set_minimum_size (w, h)
			main_window.viewport.set_item_size (w, h)
		end

end

--|--------------------------------------------------------
--| This file is Copyright (C) 2004 by ETH Zurich.
--|
--| For questions, comments, additions or suggestions on
--| how to improve this package, please write to:
--|
--|     Michela Pedroni <michela.pedroni@inf.ethz.ch>
--|
--|--------------------------------------------------------
